/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.bind.impl;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.LinkedList;
import java.util.List;

import org.locationtech.jts.geom.Envelope;

import nz.org.riskscape.engine.bind.BindingContext;
import nz.org.riskscape.engine.bind.TypeSafeBinder;
import nz.org.riskscape.problem.Problem;
import nz.org.riskscape.problem.ResultOrProblems;

public class EnvelopeBinder extends TypeSafeBinder<String, Envelope>  {

  public EnvelopeBinder() {
    super(String.class, Envelope.class);
  }

  // the positions of the bbox as parsed from a comma separated list of numbers
  // used when parsing the bounds parameter
  enum Ordinates {
    MIN_X, MAX_X, MIN_Y, MAX_Y;
  }

  @Override
  protected ResultOrProblems<Envelope> bindTypedValue(BindingContext context, String value) {

    String[] rawBounds = value.split(",");
    List<Double> parsed = new ArrayList<>(rawBounds.length);

    List<Problem> problems = new LinkedList<>();
    for (int i = 0; i < rawBounds.length; i++) {
      String ordinateString = rawBounds[i].trim();
      try {
        double ordinateValue = Double.parseDouble(ordinateString);
        parsed.add(ordinateValue);
      } catch (NumberFormatException ex) {
        problems.add(Problem.error("%s", String.format(
            "Ordinate %d (%s) is not a number - %s",
            i + 1, ordinateString, ex.getMessage())));
      }
    }

    if (rawBounds.length != Ordinates.values().length) {
      problems.add(Problem.error("%s", String.format(
          "A bounding box must consist of exactly 4 numbers - (%s) - was %d",
          Arrays.asList(Ordinates.values()),
          rawBounds.length)));
    }

    if (problems.isEmpty()) {
      return ResultOrProblems.of(new Envelope(
          parsed.get(Ordinates.MIN_X.ordinal()),
          parsed.get(Ordinates.MAX_X.ordinal()),
          parsed.get(Ordinates.MIN_Y.ordinal()),
          parsed.get(Ordinates.MAX_Y.ordinal())
        ));
    } else {
      return ResultOrProblems.failed(problems);
    }
  }

}
