/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.bind.impl;

import java.net.URI;
import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;
import java.util.Optional;
import java.util.Set;

import nz.org.riskscape.engine.data.coverage.GridTypedCoverage;
import org.geotools.api.coverage.grid.GridCoverage;

import com.google.common.collect.ImmutableSet;

import nz.org.riskscape.engine.bind.BaseBinder;
import nz.org.riskscape.engine.bind.BindingContext;
import nz.org.riskscape.engine.coverage.TypedCoverage;
import nz.org.riskscape.engine.data.Bookmark;
import nz.org.riskscape.engine.data.BookmarkFactory;
import nz.org.riskscape.engine.data.BookmarkProblems;
import nz.org.riskscape.engine.data.ResolvedBookmark;
import nz.org.riskscape.engine.problem.ProblemFactory;
import nz.org.riskscape.engine.problem.SeverityLevel;
import nz.org.riskscape.engine.relation.Relation;
import nz.org.riskscape.engine.types.WithMetadata;
import nz.org.riskscape.problem.Problem;
import nz.org.riskscape.problem.Problem.Severity;
import nz.org.riskscape.problem.Problems;
import nz.org.riskscape.problem.ResultOrProblems;
import nz.org.riskscape.rl.TokenTypes;

public class BookmarkBinder extends BaseBinder {

  public static final Set<Class<?>> SUPPORTED_TYPES = ImmutableSet.of(
      Relation.class,
      GridCoverage.class,
      GridTypedCoverage.class,
      ResolvedBookmark.class
  );

  interface LocalProblems extends ProblemFactory {

    @SeverityLevel(Severity.WARNING)
    Problem couldNotConvert();

  }

  public static final LocalProblems PROBLEMS = Problems.get(LocalProblems.class);


  @Override
  public boolean canBind(Class<?> sourceType, Class<?> destinationType) {
    return sourceType.equals(String.class) && SUPPORTED_TYPES.contains(destinationType);
  }

  @Override
  public <T> ResultOrProblems<T> bindValue(BindingContext context, Object obj, Class<T> destinationType) {
    String value = TokenTypes.stripQuotes(obj.toString());

    ResultOrProblems<Bookmark> lookupResult = wrapLookupNullOnMissing(context, Bookmark.class, value);

    if (lookupResult != null && lookupResult.hasErrors()) {
      // it exists, but it's borked, better give the user the bad news
      return ResultOrProblems.failed(lookupResult.getProblems());
    }

    Bookmark bookmark;

    if (lookupResult == null) {
      // check in case it's a bookmark('foo', ...) expression
      bookmark = getBookmarkFromExpression(context, value).orElse(null);
    } else {
      bookmark = lookupResult.get();
    }

    if (bookmark == null) {
      // this might instead be a location, try that...
      URI tryThis = BookmarkFactory.uriFromLocation(value, context.getRelativeTo()).orElse(null);

      if (tryThis == null) {
        // this typically only happens if the value couldn't be turned in to a URI, in which case we give up
        return ResultOrProblems.failed(BookmarkProblems.get().notBookmarkOrFile(value));
      } else {
        // we now have a bookmark built from a URI, let's try and resolve it...
        bookmark = Bookmark.fromURI(tryThis, value);
      }
    }

    boolean keepWrapped;
    Class<?> requiredType;
    if (ResolvedBookmark.class.isAssignableFrom(destinationType)) {
      keepWrapped = true;
      requiredType = Object.class;
    } else {
      keepWrapped = false;
      requiredType = destinationType;
    }

    // accept any type for now - we are open to some jujzing, so validate everything but the type - we will check the
    // requiredType matches further down the road
    ResultOrProblems<ResolvedBookmark> resolvedOr =
        context.getEngine().getBookmarkResolvers().resolveAndValidate(bookmark, context, Object.class);

    // don't unwrap it - return it as is, warts and all
    if (keepWrapped) {
      // call getData() purely to reveal any problems this bookmark might have. This means
      // binding to a ResolvedBookmark produces consistent warnings with binding to a Relation
      List<Problem> problems = new ArrayList<>();
      resolvedOr.ifPresent(rb -> problems.addAll(rb.getData(rb.getDataType()).getProblems()));

      return resolvedOr.map(rb -> destinationType.cast(rb)).withMoreProblems(problems);
    }

    // kludge - try and convert a typed coverage to a relation

    List<Problem> extraProblems = new LinkedList<>();
    TypedCoverage coverage = resolvedOr.flatMap(rb -> rb.getData(TypedCoverage.class)).orElse(null);
    if (requiredType.equals(Relation.class) && coverage != null) {
      Relation relation = coverage.asRelation().orElse(null);

      if (relation != null) {
        return ResultOrProblems.of(destinationType.cast(relation));
      } else {
        // let the user know we tried
        extraProblems.add(PROBLEMS.couldNotConvert());
      }
    }

    return resolvedOr.flatMap(rb -> rb.getData(destinationType).withMoreProblems(extraProblems));
  }

  /**
   * Tries to realize the given value as a riskscape expression. If it's a valid
   * bookmark() expression, then we're able to pluck out the bookmark metatdata
   * that was used in the realization
   *
   * @return bookmark produced by the given expression, or Optional.empty() if
   *         it's not a valid bookmark expression
   */
  private Optional<Bookmark> getBookmarkFromExpression(BindingContext context, String value) {
    return context.getRealizationContext().realizeConstant(value)
      .map(realized -> WithMetadata.find(realized.getResultType(), Bookmark.class))
      .orElse(Optional.empty());
  }
}
