/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.bind;

import java.util.Arrays;
import java.util.List;

import lombok.Getter;
import lombok.RequiredArgsConstructor;

/**
 * Describes aspects of how the UI should present the parameter to the user,
 * i.e. what type of input field should be used on the form.
 */
@RequiredArgsConstructor
public enum InputFieldProperty implements ParameterProperty {

  /**
   * Hide this parameter by default, i.e. it's expert-only
   */
  HIDDEN,

  /**
   * Display this parameter's value, but don't let the user edit it by default
   */
  READONLY,

  /**
   * Specify explicitly that we don't want the default formatting for numbers.
   * Useful for things like years, where we want to see 2024 rather than 2,024
   */
  UNFORMATTED,

  /**
   * User can specify multiple values, either from a pre-defined set of choices or as free-form numbers
   */
  MULTISELECT(TypedProperty.LIST),

  /**
   * Same as multiselect, but on the UI it appears as a set of checkboxes. This
   * requires that options are defined for the template
   */
  CHECKBOX(TypedProperty.LIST),

  /**
   * Specify explicitly that the UI should be a drop-down select box. Normally
   * this is inferred when there are options.
   */
  DROPDOWN,

  /**
   * Specify explicitly that the UI should be a simple textbox, i.e. don't go inferring any
   * fancy numeric multiselects, just let the user enter things in plain text
   */
  TEXTBOX;

  /**
   * Most InputFieldProperties don't make sense together (e.g. CHECKBOX and dropdown).
   * These properties can be stacked with others with sensible results.
   */
  private static final List<ParameterProperty> STACKABLE = List.of(READONLY, HIDDEN, UNFORMATTED);

  @Getter
  private final List<ParameterProperty> implied;

  InputFieldProperty(ParameterProperty... implied) {
    this(Arrays.asList(implied));
  }

  @Override
  public boolean isCompatible(ParameterProperty other) {
    // Do these properties work together
    if (STACKABLE.contains(this) || STACKABLE.contains(other)) {
      return true;
    }
    // Otherwise it only makes sense to specify one of these input form controls
    return !(other instanceof InputFieldProperty && !this.equals(other));
  }

  @Override
  public String getKeyword() {
    return name().toLowerCase();
  }
}

