/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */

package nz.org.riskscape.engine;

import java.net.URI;
import java.nio.file.Paths;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.function.Consumer;
import java.util.function.Function;

import org.geotools.referencing.crs.DefaultGeographicCRS;
import org.geotools.api.referencing.crs.CoordinateReferenceSystem;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.collect.Sets;
import com.google.common.util.concurrent.ExecutionError;
import com.google.common.util.concurrent.UncheckedExecutionException;

import lombok.Getter;
import lombok.RequiredArgsConstructor;
import lombok.Setter;

import nz.org.riskscape.config.Config;
import nz.org.riskscape.engine.bind.BindingContext;
import nz.org.riskscape.engine.bind.DefaultBindingContext;
import nz.org.riskscape.engine.bind.UserDefinedParameter;
import nz.org.riskscape.engine.data.Bookmark;
import nz.org.riskscape.engine.data.Bookmarks;
import nz.org.riskscape.engine.data.DefaultBookmarks;
import nz.org.riskscape.engine.function.FunctionResolver;
import nz.org.riskscape.engine.function.IdentifiedFunction;
import nz.org.riskscape.engine.geo.GeometryValidation;
import nz.org.riskscape.engine.model.IdentifiedModel;
import nz.org.riskscape.engine.rl.DefaultExpressionRealizer;
import nz.org.riskscape.engine.rl.ExpressionRealizer;
import nz.org.riskscape.engine.rl.RealizationContext;
import nz.org.riskscape.engine.types.Nullable;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Struct.StructMember;
import nz.org.riskscape.engine.typeset.TypeSet;
import nz.org.riskscape.engine.typexp.DefaultTypeBuilder;
import nz.org.riskscape.engine.typexp.TypeBuilder;
import nz.org.riskscape.problem.Problem;
import nz.org.riskscape.problem.ProblemSink;
import nz.org.riskscape.problem.Problems;

public class DefaultProject extends DefaultIdentifiedLocator implements Project {

  @Getter
  private final Engine engine;

  @Getter
  private final SRIDSet sridSet;

  @Getter
  private Config config;

  @Getter @Setter
  private CoordinateReferenceSystem defaultCrs = DefaultGeographicCRS.WGS84;

  @Setter
  private URI relativeTo;

  @Setter
  private URI outputBaseLocation;

  @Getter
  private GeometryValidation geometryValidation;

  @Getter
  private final Bookmarks bookmarks = put(new DefaultBookmarks());

  @Getter
  private final FunctionSet functionSet = put(new DefaultFunctionSet());

  @Getter
  private final TypeSet typeSet;

  @Getter
  private final IdentifiedCollection<IdentifiedModel> identifiedModels =
    put(new IdentifiedCollection.Base<>(IdentifiedModel.class));

  @Getter
  private final IdentifiedCollection<UserDefinedParameter> parameterTemplates =
    put(new IdentifiedCollection.Base<>(UserDefinedParameter.class));

  private TypeBuilder typeBuilder;

  public DefaultProject(Engine engine, Config config) {
    this.engine = engine;
    this.typeSet = put(new TypeSet(engine.getTypeRegistry()));
    this.sridSet = new SRIDSet(getProblemSink());
    this.config = config;
    // default geometry validation to fix automatically or bail if can't fix
    setGeometryValidation(GeometryValidation.ERROR);
  }

  @Override
  public void validate(Consumer<Problem> problemConsumer) {
    BindingContext context = newBindingContext();
    typeSet.validate(problemConsumer);
    functionSet.validate(context, problemConsumer);
    bookmarks.validate(this, problemConsumer);
  }

  @Override
  public BindingContext newBindingContext() {
    return newBindingContext(newRealizationContext());
  }

  @Override
  public BindingContext newBindingContext(RealizationContext realizationContext) {
    return new DefaultBindingContext(this, realizationContext);
  }

  @Override
  public void add(Bookmark bookmark) {
    bookmarks.add(bookmark);
  }

  @Override
  public void add(IdentifiedFunction function) {
    functionSet.add(function);
  }

  @Override
  public URI getOutputBaseLocation() {
    if (outputBaseLocation != null) {
      return outputBaseLocation;
    }
    return getRelativeTo().resolve("output/");
  }

  @Override
  public TypeBuilder getTypeBuilder() {
    if (typeBuilder == null) {
      typeBuilder = new DefaultTypeBuilder(getTypeSet());
    }

    return typeBuilder;
  }

  // puts a new equality method around a struct to allow some special case behaviour around what constitutes
  // an equal struct for the purposes of normalization.  That special case is that the types of members themselves
  // must be the same objects, not just equal.  Without this, child structs can be substituted out, which can break
  // realized expressions  (see Tuple#checkOwner) that construct structs/tuples.
  @RequiredArgsConstructor
  private static class RekeyedStruct {
    private final Struct source;

    @Override
    public boolean equals(Object obj) {
      if (obj instanceof RekeyedStruct) {
        RekeyedStruct rhs = (RekeyedStruct) obj;
        Iterator<StructMember> lhsMembers = this.source.getMembers().iterator();
        Iterator<StructMember> rhsMembers = rhs.source.getMembers().iterator();

        while (lhsMembers.hasNext()) {
          if (!rhsMembers.hasNext()) {
            return false;
          }

          StructMember lhsMember = lhsMembers.next();
          StructMember rhsMember = rhsMembers.next();

          if (!lhsMember.getKey().equals(rhsMember.getKey())) {
            return false;
          }

          if (Nullable.is(lhsMember.getType()) !=  Nullable.is(rhsMember.getType())) {
            // only one side is nullable, cannot be the same.
            return false;
          }
          // We strip nullable and check that the contained type is the same instance, and don't worry about
          // the nullable being a different instance - it won't affect the results of this method in a deleterious way
          if (Nullable.strip(lhsMember.getType()) != Nullable.strip(rhsMember.getType())) {
            return false;
          }
        }

        // extras on the rhs - not so good, al
        return !rhsMembers.hasNext();
      } else {
        return false;
      }
    }

    @Override
    public int hashCode() {
      return source.hashCode();
    }
  }

  @Override
  public RealizationContext newRealizationContext() {
    FunctionResolver functionResolver = new DefaultFunctionResolver();
    return new RealizationContext() {

      ExpressionRealizer realizer = new DefaultExpressionRealizer(this);

      Cache<Object, Object> cache = CacheBuilder.<Object,Object>newBuilder()
          .softValues()
          // we use Math.max to ensure the concurrency level is set to something valid
          // this is mainly to keep tests passing because maybe they don't have a real pipeline executor
          .concurrencyLevel(Math.max(1, engine.getPipelineExecutor().getNumThreads()))
          .build();

      /**
       * Maintains a mapping of normalized structs, to avoid object proliferation.
       * When two different structs are equal (i.e. Object#equals), then when
       * normalized they end up as the exact same object (i.e. Object ==). The first
       * struct normalized populates the map, then subsequent key lookups of
       * equivalent structs (Object#equals) will return the first canonical struct.
       */
      HashMap<RekeyedStruct, Struct> structs = new HashMap<>();
      // add this one in - reduces the need for tests etc to normalize it themselves
      Struct empty = normalizeStruct(Struct.EMPTY_STRUCT);

      @Override
      public ExpressionRealizer getExpressionRealizer() {
        return realizer;
      }

      @Override
      public Project getProject() {
        return DefaultProject.this;
      }

      @Override
      public FunctionResolver getFunctionResolver() {
        return functionResolver;
      }

      @Override
      public Struct normalizeStruct(Struct struct) {
        // normalizing should typically only be done during realization, which is single-threaded.
        // However, dynamically loading a relation bookmark can do projection/struct realizing
        // on the fly, and this can occur during multi-threaded pipeline execution
        synchronized (this) {
          return structs.computeIfAbsent(new RekeyedStruct(struct), key -> key.source);
        }
      }

      @Override
      public ProblemSink getProblemSink() {
        return engine.getProblemSink();
      }

      @Override
      public <T> T getOrComputeFromCache(Object cacheKey, Class<T> expectedType, Function<Object, T> compute) {
        try {
          Object computed = cache.get(cacheKey, () -> compute.apply(cacheKey));
          return expectedType.cast(computed);
        } catch (ExecutionException | UncheckedExecutionException | ExecutionError e) {
          // These exceptions are thrown by the Cache if it catches an exception whilst loading a cache value.
          // All of these exceptions wrap the exception that trigged it.
          Throwable cause = e.getCause();
          if (cause instanceof RiskscapeException) {
            throw (RiskscapeException)cause;
          }
          throw new RiskscapeException(Problems.caught(e.getCause()));
        }
      }
    };
  }

  @Override
  public URI getRelativeTo() {
    if (relativeTo == null) {
      // if relativeTo has not been set we should be relative to the current working directory.
      return Paths.get("").toUri();
    }
    return relativeTo;
  }

  @Override
  public boolean hasCollectionOf(Class<? extends Identified> identifiedClass) {
    return super.hasCollectionOf(identifiedClass) || getEngine().hasCollectionOf(identifiedClass);
  }

  @Override
  public <T extends Identified> IdentifiedCollection<T> getCollectionByClass(Class<T> collectionClass) {

    if (engine.hasCollectionOf(collectionClass)) {
      return engine.getCollectionByClass(collectionClass);
    }

    return super.getCollectionByClass(collectionClass);
  }

  @Override
  public Set<Class<? extends Identified>> getCollectionClasses() {
    return Sets.union(super.getCollectionClasses(), engine.getCollectionClasses());
  }

  @Override
  public final ProblemSink getProblemSink() {
    return engine.getProblemSink();
  }

  @Override
  public void setGeometryValidation(GeometryValidation mode) {
    geometryValidation = mode;
    sridSet.setValidationPostReproject(mode);
  }
}
