/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine;

import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

public class DefaultIdentifiedLocator implements IdentifiedLocator {

  private final Map<Class<? extends Identified>, IdentifiedCollection<?>> collections = new HashMap<>();

  @SuppressWarnings({ "unchecked", "rawtypes" })
  @Override
  public <T extends Identified> IdentifiedCollection<T> getCollectionByClass(Class<T> identifiedClass) {
    IdentifiedCollection collection = collections.get(identifiedClass);

    if (collection == null) {
      throw new IllegalArgumentException(identifiedClass + " is not registered in : " + this);
    }

    if (collection.getCollectionClass() == identifiedClass) {
      return collection;
    } else {
      throw new AssertionError("somehow a collection class has been registered with the wrong type");
    }
  }

  @Override
  public <T extends Identified, U extends IdentifiedCollection<T>> U getCollection(Class<U> collectionClass) {
    return collectionClass.cast(getCollectionByClass(findIdentifiedType(collectionClass)));
  }

  @SuppressWarnings("unchecked")
  private Class<? extends Identified> findIdentifiedType(Class<?> collectionClass) {
    List<Type> types = new ArrayList<>();
    types.add(collectionClass.getGenericSuperclass());
    types.addAll(Arrays.asList(collectionClass.getGenericInterfaces()));

    for (Type type : types) {
      if (type instanceof ParameterizedType) {
        ParameterizedType pType = (ParameterizedType) type;
        if (IdentifiedCollection.class.isAssignableFrom((Class<?>) pType.getRawType())) {
          return (Class<? extends Identified>) pType.getActualTypeArguments()[0];
        }
      }
    }

    throw new RuntimeException("Could not find parameterized type");
  }

  public void registerCollection(IdentifiedCollection<?> collection) {
    collections.put(collection.getCollectionClass(), collection);
  }

  @Override
  public boolean hasCollectionOf(Class<? extends Identified> identifiedClass) {
    return collections.containsKey(identifiedClass);
  }

  @Override
  public Set<Class<? extends Identified>> getCollectionClasses() {
    return Collections.unmodifiableSet(collections.keySet());
  }
  /**
   * Convenience method to allow extenders of this class to register collections while also assigning them to a field
   */
  protected <T extends IdentifiedCollection<?>> T put(T collection) {
    registerCollection(collection);
    return collection;
  }

  @Override
  public String toString() {
    return String.format("DefaultIdentifiedLocator(collections=%s)",
        getCollectionClasses().stream().map(Class::getSimpleName).collect(Collectors.joining(", ", "[", "]")));
  }
}
