/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.config.ini4j;

import java.util.Map;

import com.google.common.collect.ImmutableMap;

public interface Ini4jConfigConstants {

  /**
   * Auto import directory depth limit
   */
  int RECURSION_LIMIT = 64;

  /**
   * URIs matching this proto are on the file system and can be auto imported
   */
  String FILE_PROTO = "file";

  /**
   * File extension for ini files
   */
  String INI_SUFFIX = ".ini";

  /**
   * Default name of a project.ini file
   */
  String DEFAULT_PROJECT_INI = "project.ini";

  // various constants for section names and keys
  String PROJECT_SECTION = "project";
  String AUTOIMPORT_KEY = "auto-import";
  String AUTOIMPORT_DEFAULT = "false";
  String IMPORT_KEY = "import";

  /**
   * Maps ini file prefixes to their type prefix
   */
  Map<String, String> TYPE_PREFIXES = ImmutableMap.<String, String>builder()
      .put("types", "type")
      .put("bookmarks", "bookmark")
      .put("models", "model")
      .put("functions", "function")
      .put("parameters", "parameter")
      .build();

  /**
   * @return the implicit type prefix for the given filename, or null if none exists.  Sections without a well known
   * type prefix in an ini file with this name should be given the resulting prefix.  For examples, a section like
   * `[cool section]` in a file named `types.ini` would be renamed to be like `[type cool section]`
   */
  default String getWellKnownTypePrefix(String filename) {
    for (String fileprefix : Ini4jConfigSection.TYPE_PREFIXES.keySet()) {
      if (filename.startsWith(fileprefix)) {
        return Ini4jConfigSection.TYPE_PREFIXES.get(fileprefix);
      }
    }

    return null;
  }
}
