/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.config.ini;

import java.net.URI;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import lombok.Getter;
import nz.org.riskscape.config.ConfigSection;
import nz.org.riskscape.config.ConfigString;
import nz.org.riskscape.dsl.Token;
import nz.org.riskscape.ini.KeyValue;
import nz.org.riskscape.ini.Section;

public class IniConfigSection extends ConfigSection {

  public IniConfigSection(URI location, String name, Section section) {
    super(section.getHeaderToken().getLocation().addToUri(location), name);
    this.section = section;

    configMap = section.getMembers().stream().collect(Collectors.toMap(
        kv -> ConfigString.wrap(location, kv.getKey()),
        // don't add an empty string to the list (add a unit test for this)
        kv -> kv.isBlank() ? List.of() : List.of(encodeValue(kv)),
        (a, b) -> {
          List<ConfigString> list = new ArrayList<>(a.size() + b.size());
          list.addAll(a);
          list.addAll(b);
          return list;
        },
        LinkedHashMap::new
    ));
  }

  @Getter
  private final Section section;
  private final Map<ConfigString, List<ConfigString>> configMap;

  @Override
  public Map<ConfigString, List<ConfigString>> toConfigMap() {
    return configMap;
  }

  private ConfigString encodeValue(KeyValue keyValue) {
    Token token = keyValue.getValueToken();
    // use keyValue#getValue() as that takes care of extraneous whitespace for us
    return new ConfigString.Wrapper(keyValue.getValue()) {
      @Override
      public URI getLocation() {
        return token.getLocation().addToUri(IniConfigSection.this.getLocation());
      }
    };
  }
}
