/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.config;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import java.util.function.Predicate;
import java.util.stream.Stream;

import nz.org.riskscape.engine.Project;
import nz.org.riskscape.engine.resource.Resource;
import nz.org.riskscape.engine.resource.ResourceLoadingException;
import nz.org.riskscape.engine.resource.UriHelper;
import nz.org.riskscape.engine.util.Pair;
import nz.org.riskscape.problem.Problem;
import nz.org.riskscape.problem.Problems;

public class ConfigUtils {


  public static Stream<Resource> getResources(
      Project project,
      Consumer<Problem> problemSink,
      Predicate<ConfigSection> fromSections,
      String key
  ) {
    return project.getConfig().getAll().stream()
      // get all the relevant sections
      .filter(fromSections)
      .flatMap(cs -> {
        // convert all the functions to uris relative to the config section
        List<String> values = cs.getAll(key);
        return (values == null ? Collections.<String>emptyList() : values).stream()
          .map(str -> Pair.of(cs, UriHelper.uriFromLocation(str, cs.getLocation())));
      }
      )
    .flatMap(configSectionAnduriOr ->
      // load the function, or report an error
      configSectionAnduriOr.getRight().ifElseReturn(
        uri -> {
          try {
            Resource resource = project.getEngine().getResourceFactory().load(uri);
            return Collections.singletonList(resource).stream();
          } catch (ResourceLoadingException rex) {
            problemSink.accept(Problems.foundWith(configSectionAnduriOr.getLeft(), rex.getProblem()));
            return Collections.<Resource>emptyList().stream();
          }
        },
        probs -> {
          problemSink.accept(Problems.foundWith(configSectionAnduriOr.getLeft(), probs));
          return Collections.<Resource>emptyList().stream();
        }
      )
     );
  }

}
