/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.config;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.function.Function;

import lombok.Getter;
import lombok.RequiredArgsConstructor;
import nz.org.riskscape.engine.bind.ParamProblems;
import nz.org.riskscape.engine.bind.Parameter;
import nz.org.riskscape.engine.bind.impl.BooleanBinder;
import nz.org.riskscape.engine.bind.impl.NumberBinder;
import nz.org.riskscape.problem.Problem.Severity;
import nz.org.riskscape.problem.ProblemSink;
import nz.org.riskscape.problem.Problems;
import nz.org.riskscape.problem.ResultOrProblems;

/**
 * Used to load minimal INI config settings, whilst the engine is still bootstrapping,
 * i.e. we don't have any BindingContext available yet. Loading RiskScape's settings.ini file
 * is one example, as this needs to be done first, before any project.ini is loaded.
 */
@RequiredArgsConstructor
public class BootstrapIniSettings {

  public static final BootstrapIniSettings NONE = new BootstrapIniSettings(Collections.emptyMap(), ProblemSink.DEVNULL);

  @Getter
  private final Map<String, List<String>> unbound;

  private final ProblemSink sink;

  /**
   * Extracts a single, numeric value from the settings.ini config for the given parameter
   */
  public <T extends Number> T getOrDefault(String parameterName, T defaultValue) {
    return getOrDefault(parameterName, defaultValue,
        v -> (ResultOrProblems<T>) new NumberBinder().bindValue(v, defaultValue.getClass()));
  }

  /**
   * Extracts a single, boolean value from the settings.ini config for the given parameter
   */
  public boolean getOrDefault(String parameterName, boolean defaultValue) {
    return getOrDefault(parameterName, defaultValue, v -> new BooleanBinder().bindValue(v));
  }

  private <T> T getOrDefault(String parameterName, T defaultValue, Function<Object, ResultOrProblems<T>> binder) {
    List<String> values = unbound.getOrDefault(parameterName, Collections.emptyList());
    if (values.isEmpty()) {
      return defaultValue;
    }

    if (values.size() != 1) {
      sink.log(ParamProblems.get().wrongNumberGiven(parameterName, "1", values.size()).withSeverity(Severity.WARNING));
    }

    ResultOrProblems<T> bound = binder.apply(values.get(0));
    bound.getProblems().forEach(p -> sink.log(
        Problems.foundWith(Parameter.class, parameterName, p.withSeverity(Severity.WARNING))
    ));

    return bound.hasErrors() ? defaultValue : (T) bound.getWithProblemsIgnored();
  }
}
