/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.output;

import java.util.Optional;
import java.util.function.Consumer;

import com.google.common.collect.Lists;

import lombok.Data;
import lombok.EqualsAndHashCode;
import nz.org.riskscape.engine.bind.ParameterField;
import nz.org.riskscape.problem.ResultOrProblems;
import nz.org.riskscape.rl.ast.Expression;

/**
 * A test format that accepts options. The writer produced by this format is a standard {@link CsvWriter}.
 *
 * Whenever a writer is constructed via the {@link WriterConstructor} any options given will be
 * passed to the optionConsumer.
 */
public class TestFormat extends BaseFormat {

  @EqualsAndHashCode @Data
  public static class TestFormatOptions extends FormatOptions {
    @ParameterField
    public String required;
    @ParameterField
    public int number = 10;
    @ParameterField
    public Optional<Expression> expression = Optional.empty();
  }

  private final Consumer<Optional<?>> optionConsumer;

  public TestFormat(Consumer<Optional<?>> optionComsumer, Characteristics... extraCharacteristics) {
    super(
        "test", "csv", "text/csv", Characteristics.STREAMABLE,
        Lists.asList(Characteristics.NON_SPATIAL, extraCharacteristics).toArray((new Characteristics[]{}))
    );

    this.optionConsumer = optionComsumer;
  }

  @Override
  public Optional<WriterConstructor> getWriterConstructor() {
    return Optional.of((context, type, handle, options) -> {
      // we stash the options away so tests can verify they are expected.
      // we'd expect an actual format that supports options to pass them into the writer
      optionConsumer.accept(options);
      return ResultOrProblems.of(new CsvWriter(type, context.getProject().getProblemSink(), handle, true));
    });
  }

  @Override
  public Class<? extends FormatOptions> getWriterOptionsClass() {
    return TestFormatOptions.class;
  }

}
