/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.util;

import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import java.util.Iterator;
import java.util.Map;
import java.util.stream.LongStream;

import org.junit.Test;

import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Types;

public class SegmentedTupleListTest {

  @Test
  public void testAnEmptyList() {
    SegmentedTupleList list = new SegmentedTupleList(Types.TEXT.asStruct());

    assertThat(list, hasSize(0));
    assertThat(list.iterator().hasNext(), is(false));
  }

  @Test
  public void testBuildAListFromALotOfTuples() throws Exception {
    // big enough that our underlying lists are segmented
    final int numTuples = 100;
    Struct struct = Struct.of("foo", Types.INTEGER, "bar", Types.TEXT);

    SegmentedTupleList list = new SegmentedTupleList(struct);

    LongStream.range(0, numTuples)
      .mapToObj(number -> Tuple.ofValues(struct, number, "text" + number))
      .forEach(list::add);

    assertThat(list, hasSize(numTuples));

    // check iteration
    Iterator<Tuple> iterator = list.iterator();
    for (int i = 0; i < numTuples; i++) {
      Tuple tuple = iterator.next();
      assertThat(tuple.getStruct(), sameInstance(struct));
      assertThat(tuple.fetch("foo"), equalTo((long) i));
      assertThat(tuple.fetch("bar"), equalTo("text" + i));
    }

    assertThat(iterator.hasNext(), is(false));

    // check random access
    assertThat(list.get(0), equalTo(Tuple.ofValues(struct, 0L, "text0")));
    assertThat(list.get(55), equalTo(Tuple.ofValues(struct, 55L, "text55")));
    assertThat(list.get(99), equalTo(Tuple.ofValues(struct, 99L, "text99")));
  }

  @Test
  public void tuplesCanBeNested() throws Exception {
    Struct struct = Struct.of("foo", Types.INTEGER, "bar", Struct.of("baz", Types.TEXT));

    SegmentedTupleList list = new SegmentedTupleList(struct);
    list.add(Tuple.ofValues(struct, 1L, Tuple.ofValues((Struct) struct.getEntry("bar").getType(), "qux")));

    Tuple next = list.iterator().next();
    assertThat(next.toMap(), equalTo(Map.of("foo", 1L, "bar", Map.of("baz", "qux"))));
  }

  @Test
  public void canAddAllOfAnotherList() throws Exception {
    final int numTuples = 100;
    Struct struct = Struct.of("foo", Types.INTEGER);

    SegmentedTupleList listA = new SegmentedTupleList(struct);
    SegmentedTupleList listB = new SegmentedTupleList(struct);

    LongStream.range(0, numTuples).mapToObj(number -> Tuple.ofValues(struct, number)).forEach(listA::add);
    LongStream.range(numTuples, numTuples * 2).mapToObj(number -> Tuple.ofValues(struct, number)).forEach(listB::add);

    listA.addAll(listB);

    // a is bigger, b is the same
    assertThat(listA, hasSize(numTuples * 2));
    assertThat(listB, hasSize(numTuples));


    Iterator<Tuple> iterator = listA.iterator();
    for (int i = 0; i < numTuples * 2; i++) {
      Tuple tuple = iterator.next();
      assertThat(tuple.getStruct(), sameInstance(struct));
      assertThat(tuple.fetch("foo"), equalTo((long) i));
    }

    assertThat(iterator.hasNext(), is(false));
  }
}
