/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.util;

import static nz.org.riskscape.engine.FileSystemMatchers.*;
import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import java.io.BufferedReader;
import java.io.FileReader;
import java.nio.file.Path;
import java.util.List;
import java.util.stream.Collectors;

import org.junit.After;
import org.junit.Before;
import org.junit.Test;

import com.codahale.metrics.MetricRegistry;

import nz.org.riskscape.engine.TemporaryDirectoryTestHelper;
import nz.org.riskscape.problem.ProblemSink;

public class FileProgressNotifierTest implements TemporaryDirectoryTestHelper {

  // the tests will drive writing to file. A large delay should guarantee we
  // don't write to the file via timer in the background
  private static final long ONE_HOUR_DELAY = 60 * 60 * 1000;

  MetricRegistry metrics = new MetricRegistry();
  Path tempDir;
  long fooValue = 0;

  @Before
  public void setupTempDir() throws Exception {
    tempDir = createTempDirectory("file-progress");
  }

  @After
  public void cleanupTempDir() throws Exception {
    remove(tempDir);
  }

  private List<String> readContents(Path path) throws Exception {
    try (FileReader reader = new FileReader(path.toFile())) {
      return new BufferedReader(reader).lines()
          .filter(s -> !s.trim().isEmpty())
          .collect(Collectors.toList());
    }
  }

  @Test
  public void canOutputProgressToFile() throws Exception {
    Path progressFile = tempDir.resolve("progress.txt");
    FileProgressNotifier monitor = new FileProgressNotifier(metrics, progressFile, ProblemSink.DEVNULL, ONE_HOUR_DELAY,
        ONE_HOUR_DELAY);

    // no metrics should result in empty file
    monitor.updateStatus();
    assertThat(tempDir, hasFile(fileWithName("progress.txt")));
    assertThat(readContents(progressFile), empty());

    PercentGauge fooGauge = new PercentGauge(() -> fooValue, () -> 100L);
    fooValue = 50;
    metrics.register("foo", fooGauge);

    monitor.updateStatus();
    assertThat(tempDir, hasFile(fileWithName("progress.txt")));
    assertThat(readContents(progressFile), contains(
        allOf(containsString("foo"), containsString("50.00%"))
    ));

    fooValue = 75;
    metrics.register("bar", new PercentGauge(() -> 25L, () -> 100L));

    monitor.updateStatus();
    assertThat(tempDir, hasFile(fileWithName("progress.txt")));
    assertThat(readContents(progressFile), containsInAnyOrder(
        allOf(containsString("foo"), containsString("75.00%")),
        allOf(containsString("bar"), containsString("25.00%"))
    ));

    monitor.finish();

    assertThat(tempDir, hasFile(fileWithName("progress.txt")));
    assertThat(readContents(progressFile), containsInAnyOrder(
        allOf(containsString("foo"), containsString("75.00%")),
        allOf(containsString("bar"), containsString("25.00%")),
        containsString("Complete!"))
    );
  }
}
