/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.relation;

import static nz.org.riskscape.engine.Assert.*;
import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import org.junit.Test;

import nz.org.riskscape.engine.RiskscapeIOException;
import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.types.DuplicateKeysException;
import nz.org.riskscape.engine.types.Nullable;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Types;

public class ZipRelationTest {

  ZipRelation relation;

  ListRelation wholeNumbers = ListRelation.ofTypedValues(Struct.of("foo", Types.INTEGER), 1L, 2L, 3L);
  ListRelation textStrings = ListRelation.ofTypedValues(Struct.of("bar", Types.TEXT), "one", "two", "three");
  ListRelation structs = ListRelation.ofTypedValues(
      Struct.of("baz", Types.FLOATING, "qux", Nullable.TEXT), 1.0D, 2.0D, 3.0D);

  @Test
  public void canBuildZipRelation() throws Exception {
    relation = new ZipRelation(wholeNumbers, textStrings);
    Struct expected = wholeNumbers.getType().and(textStrings.getType());
    assertThat(relation.getType(), is(Struct.of("foo", Types.INTEGER, "bar", Types.TEXT)));
    assertThat(relation.size().get(), is(3L));

    TupleIterator iterator = relation.iterator();
    assertTrue(iterator.hasNext());
    assertThat(iterator.next(), is(Tuple.ofValues(expected, 1L, "one")));
    assertTrue(iterator.hasNext());
    assertThat(iterator.next(), is(Tuple.ofValues(expected, 2L, "two")));
    assertTrue(iterator.hasNext());
    assertThat(iterator.next(), is(Tuple.ofValues(expected, 3L, "three")));
    assertFalse(iterator.hasNext());

    // new iterator starts again from scratch
    assertTrue(relation.iterator().hasNext());
  }

  @Test
  public void canFlattenStructRelations() throws Exception {
    relation = new ZipRelation(wholeNumbers, structs);
    // it should work the same way as joins - all members get dumped into one big struct
    assertThat(relation.getType(), is(Struct.of(
        "foo", Types.INTEGER,
        "baz", Types.FLOATING,
        "qux", Nullable.TEXT
    )));
  }

  @Test
  public void cannotDuplicateKeysBetweenRelations() throws Exception {
    DuplicateKeysException ex = assertThrows(DuplicateKeysException.class,
        () -> new ZipRelation(wholeNumbers, (BaseRelation) wholeNumbers.limit(3)));
  }

  @Test
  public void willGetExceptionWhenRelationsDifferInSize() throws Exception {
    RiskscapeIOException ex = assertThrows(RiskscapeIOException.class,
        () -> new ZipRelation(wholeNumbers, ListRelation.ofValues("missing a few...")));
    assertThat(ex.getProblem(), is(ZipRelation.PROBLEMS.mismatchingRows()));
  }
}
