/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.relation;

import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;

import org.junit.Test;

import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Types;

public class UnionRelationTest {

  UnionRelation relation;

  @Test
  public void returnsEmptyIfListsAreEmpty() throws Exception {
    relation = new UnionRelation(Struct.of(), Arrays.asList(
        new EmptyRelation(Struct.EMPTY_STRUCT),
        new EmptyRelation(Struct.EMPTY_STRUCT)
    ));
    assertFalse(iterator().hasNext());

    relation = new UnionRelation(Struct.of(), Arrays.asList());
    assertFalse(iterator().hasNext());
  }

  @Test
  public void returnsAnIteratorForASingleRelation() throws Exception {
    relation = new UnionRelation(ListRelation.ofValues(1L, 2L, 3L));
    assertThat(collectValues(), contains(1L, 2L, 3L));
  }

  @Test
  public void willCombineTheResultsOfTwoRelations() throws Exception {
    relation = new UnionRelation(
        ListRelation.ofValues(1L, 2L, 3L),
        ListRelation.ofValues(7L, 9L, 8L));

    assertThat(collectValues(), contains(1L, 2L, 3L, 7L, 9L, 8L));
  }

  @Test
  public void willJoinTheResultsOfManyRelationsOfVariousSizes() throws Exception {
    relation = new UnionRelation(
        ListRelation.ofValues("foo"),
        ListRelation.ofValues("bar", "baz"),
        ListRelation.ofValues("more", "foo"),
        ListRelation.ofValues("a", "lot", "of", "baz")
    );

    assertThat(collectValues(), contains("foo", "bar", "baz", "more", "foo", "a", "lot", "of", "baz"));
  }

  @Test
  public void willSkipAFirstEmptyRelation() throws Exception {
    relation = new UnionRelation(
        new ListRelation(Types.TEXT.asStruct()),
        ListRelation.ofValues("foo", "bar", "baz")
    );

    assertThat(collectValues(), contains("foo", "bar", "baz"));
  }

  @Test
  public void willSkipAnIntermediateEmptyRelation() throws Exception {
    relation = new UnionRelation(
        ListRelation.ofValues("foo"),
        new ListRelation(Types.TEXT.asStruct()),
        ListRelation.ofValues("bar", "baz")
    );

    assertThat(collectValues(), contains("foo", "bar", "baz"));
  }

  @Test
  public void willSkipAFinalEmptyRelation() throws Exception {
    relation = new UnionRelation(
        ListRelation.ofValues("foo"),
        ListRelation.ofValues("bar", "baz"),
        new ListRelation(Types.TEXT.asStruct())
    );

    assertThat(collectValues(), contains("foo", "bar", "baz"));
  }

  private TupleIterator iterator() {
    return relation.iterator();
  }

  private List<?> collectValues() {
    return relation.stream().map(t -> t.fetch("value")).collect(Collectors.toList());
  }



}
