/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.relation;

import static org.junit.Assert.*;

import java.util.Collections;
import java.util.List;
import java.util.function.Predicate;

import org.junit.Test;

import nz.org.riskscape.engine.Matchers;
import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.restriction.Restriction;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.problem.Problem;

public class BaseRelationLimitOffsetTest {

  ListRelation source = ListRelation.ofValues(1L, 2L, 3L, 4L, 5L, 6L, 7L, 8L);

  @Test
  public void canApplyALimitToARelation() {
    assertThat(source.limit(5), Matchers.relationWithValues(1L, 2L, 3L, 4L, 5L));
  }


  @Test
  public void canApplyAnOffsetToARelation() throws Exception {
    assertThat(source.offset(3), Matchers.relationWithValues(4L, 5L, 6L, 7L, 8L));
  }

  @Test
  public void canApplyALimitAndOffsetToARelation() {
    assertThat(source.limitAndOffset(3, 3), Matchers.relationWithValues(4L, 5L, 6L));
  }

  @Test
  public void reapplyingLimitAndOffsetReplacesExistingValues() throws Exception {
    Relation a = source.limitAndOffset(3, 2);
    assertEquals(3L, a.getLimit());
    assertEquals(2L, a.getOffset());

    Relation b = a.limitAndOffset(2, 3);
    assertEquals(2L, b.getLimit());
    assertEquals(3L, b.getOffset());

    assertThat(a, Matchers.relationWithValues(3L, 4L, 5L));
    assertThat(b, Matchers.relationWithValues(4L, 5L));
  }

  @Test
  public void canApplyAnOffsetBeyondTheNumberOfTuplesInTheRelation() throws Exception {
    assertThat(source.offset(10), Matchers.relationWithValues());
  }

  @Test
  public void limitAndOffsetApplyAfterRestriction() throws Exception {
    Restriction r = new Restriction() {

      @Override
      public List<Problem> validate(Struct type) {
        return Collections.emptyList();
      }

      @Override
      public Predicate<Tuple> getPredicate() {
        return tuple -> {
          Long value = tuple.fetch("value");
          return value >= 4 && value < 9 && value != 5;
        };
      }
    };

    assertThat(source.restrict(r).get().limitAndOffset(2, 1), Matchers.relationWithValues(6L, 7L));
  }

}
