/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.function;

import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;
import static org.mockito.Mockito.*;

import java.net.URI;
import java.util.Optional;

import org.junit.Before;
import org.junit.Test;

import nz.org.riskscape.config.ConfigString;
import nz.org.riskscape.engine.Engine;
import nz.org.riskscape.engine.bind.ParamProblems;
import nz.org.riskscape.engine.resource.ResourceFactory;
import nz.org.riskscape.engine.resource.ResourceLoadingException;
import nz.org.riskscape.engine.resource.StringResource;
import nz.org.riskscape.problem.Problems;
import nz.org.riskscape.problem.ResultOrProblems;

public class FunctionFrameworkSupportTest {

  FunctionFrameworkSupport subject = new FunctionFrameworkSupport() {
  };

  Engine mockEngine = mock(Engine.class);
  ResourceFactory mockResourceFactory = mock(ResourceFactory.class);

  MetadataParams params = new MetadataParams();
  URI uri = URI.create("https://riskscape.org.nz/foo");

  @Before
  public void setup() {
    params.location = Optional.empty();
    params.source = Optional.empty();

    when(mockEngine.getResourceFactory()).thenReturn(mockResourceFactory);
  }

  @Test
  public void testLoadingFunctionFromLocation() {
    params.location = Optional.of(uri);
    StringResource resource = new StringResource(uri, "content");
    when(mockResourceFactory.load(uri)).thenReturn(resource);

    assertThat(
        subject.loadFunctionFromParams(mockEngine, params),
        equalTo(ResultOrProblems.of(resource))
    );
  }

  @Test
  public void testLoadingFunctionWithBadLocation() throws Exception {
    params.location = Optional.of(uri);
    when(mockResourceFactory.load(uri)).thenThrow(new ResourceLoadingException(uri, Problems.foundWith(uri)));

    // problem from exception is returned
    assertThat(
        subject.loadFunctionFromParams(mockEngine, params),
        equalTo(ResultOrProblems.failed(Problems.foundWith(uri)))
    );
  }

  @Test
  public void testLoadingFunctionResourceFromEmbeddedSource() {
    params.source = Optional.of(ConfigString.wrap(uri, "test"));

    assertThat(
        subject.loadFunctionFromParams(mockEngine, params),
        equalTo(ResultOrProblems.of(new StringResource(uri, "test")))
    );
  }

  @Test
  public void testOnlyOneOfSourceOrLocationIsAllowed() throws Exception {
    params.source = Optional.of(ConfigString.wrap(uri, "test"));
    params.location = Optional.of(uri);

    assertThat(
        subject.loadFunctionFromParams(mockEngine, params),
        equalTo(ResultOrProblems.failed(ParamProblems.oneOfTheseRequired("location", "source")))
    );
  }

  @Test
  public void testOneOfSourceOrLocationIsRequired() throws Exception {
    assertThat(
        subject.loadFunctionFromParams(mockEngine, params),
        equalTo(ResultOrProblems.failed(ParamProblems.oneOfTheseRequired("location", "source")))
    );
  }
}
