/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.util;

import static systems.uom.common.USCustomary.*;

import javax.measure.Quantity;
import javax.measure.Unit;
import javax.measure.format.MeasurementParseException;

import nz.org.riskscape.engine.RiskscapeException;
import nz.org.riskscape.engine.problem.ProblemFactory;
import nz.org.riskscape.problem.Problem;
import nz.org.riskscape.problem.Problems;
import tech.units.indriya.format.SimpleUnitFormat;

/**
 * Utilities for working with {@link Unit}s
 */
public class UnitUtils {

  public interface LocalProblems extends ProblemFactory {

    static LocalProblems get() {
      return Problems.get(LocalProblems.class);
    }

    Problem notRecognised(String label);

    <T extends Quantity<T>> Problem wrongType(Unit<?> found, Class<T> expected);
  }

  /**
   * Get the {@link Unit} of the required type.
   *
   * @param <T>
   * @param identifier the identifier of the desired unit
   * @param type of unit to return (length, area etc)
   * @return unit of required type
   * @throws RiskscapeException if unit is not found or is not the required type
   */
  public static <T extends Quantity<T>> Unit<T> of(String identifier, Class<T> type) throws RiskscapeException {
    try {
      Unit<?> parsed = of(identifier);
      try {
        return parsed.asType(type);
      } catch (ClassCastException e) {
        throw new RiskscapeException(LocalProblems.get().wrongType(parsed, type));
      }
    } catch (MeasurementParseException e) {
      throw new RiskscapeException(LocalProblems.get().notRecognised(identifier));
    }
  }

  private static Unit<?> of(String identifier) {
    switch(identifier) {
      case "in":
        return INCH;
      case "ft":
        return FOOT;
      case "ft_survey_us":
        return FOOT_SURVEY;
      case "mi":
        return MILE;
      case "ha":
        return HECTARE;
      default:
        // defer to simple unit format. this will work for variants of m (e.g. mm, cm, m²)
        return SimpleUnitFormat.getInstance().parse(identifier);
    }
  }

}
