/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.util;

import java.util.AbstractList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;

import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.types.Struct;

/**
 * Stores lists of tuples 'efficiently' by storing them column-wise (rather that row-wise).  This allows us to take
 * advantage of using native arrays for numeric types as well as avoiding the overhead of storing individual tuples.
 *
 * A struct like {foo: integer, bar: floating} would use around 16 bytes for the object, 4 byte pointer to the
 * object[] and a 4 byte pointer to the struct.  The array itself has an overhead of 16 bytes, and then another 8 bytes
 * for the pointers to the array elements, followed by 24 bytes for the long and 24 bytes for the double.  This gives
 * us a grand total of 96 bytes 16+4+4+16+8+48 to store our tuple with 2 values.
 *
 * Contrast that with a similar tuple stored in a SegmentedList - each tuple takes up 16 bytes, with the overhead being
 * amortized across the list segment (which will end up being < 1 byte per tuple once the list grows beyond a few
 * thousand elements).
 *
 * # Testing results
 *
 * I setup a pipeline to aggregate 40mil rows of liquefaction data, which was a uuid that keyed various statistics.  At
 * first I ran the pipeline against master, and the pipeline was about to get through about 45% of the data before
 * bogging down with constant GC.  I changed the group by to group lists of numbers, rather than a list of structs of
 * numbers and the model was able to get to 85% before bogging down in GC.
 */
final class SegmentedTupleList extends AbstractList<Tuple> {

  /**
   * The type of struct we are storing in the list
   */
  private final Struct struct;

  /**
   * Each member of the struct is itself stored in a list
   */
  private final List<? extends List<Object>> memberLists;

  SegmentedTupleList(Struct struct) {
    if (struct.size() == 0) {
      throw new IllegalArgumentException("Can not build a segmented list of empty structs");
    }

    this.struct = struct;
    this.memberLists = struct.getMembers().stream().map(member -> {
      return SegmentedList.forType(member.getType());
    }).toList();
  }

  @Override
  public boolean add(Tuple e) {
    assert e.getStruct().equals(struct);

    int index = 0;
    for (List<Object> segmentedList : memberLists) {
      segmentedList.add(e.fetch(index++));
    }

    return true;
  }

  @Override
  public void add(int index, Tuple element) {
    throw new UnsupportedOperationException();
  }

  @Override
  public Tuple get(int index) {
    return Tuple.ofValues(struct, memberLists.stream().map(list -> list.get(index)).toArray());
  }

  @Override
  public Iterator<Tuple> iterator() {
    List<Iterator<Object>> iterators = memberLists.stream().map(List::iterator).toList();

    return new Iterator<Tuple>() {

      @Override
      public Tuple next() {
        Tuple tuple = Tuple.of(struct);

        int index = 0;
        for (Iterator<Object> iterator : iterators) {
          tuple.set(index++, iterator.next());
        }

        return tuple;
      }

      @Override
      public boolean hasNext() {
        return iterators.get(0).hasNext();
      }
    };
  }

  @Override
  public boolean addAll(Collection<? extends Tuple> c) {
    if (c instanceof SegmentedTupleList stl && stl.struct == this.struct) {

      for (int i = 0; i < this.memberLists.size(); i++) {
        this.memberLists.get(i).addAll(stl.memberLists.get(i));
      }

      return true;
    } else {
      return super.addAll(c);
    }
  }

  @Override
  public int size() {
    return memberLists.get(0).size();
  }
}
