/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.util;

import com.codahale.metrics.Counter;
import com.codahale.metrics.Gauge;
import com.codahale.metrics.Histogram;
import com.codahale.metrics.Meter;
import com.codahale.metrics.MetricRegistry;
import com.codahale.metrics.MetricRegistryListener;
import com.codahale.metrics.Timer;

import lombok.RequiredArgsConstructor;

@RequiredArgsConstructor
public class ParentingListener implements MetricRegistryListener {

  /**
   * Mirrors all the metrics from source in to target, prepending each metric with prefix.  Uses this
   * {@link ParentingListener} to keep them in sync.
   */
  public static void mirror(MetricRegistry source, MetricRegistry target, String prefix) {
    source.addListener(new ParentingListener(target, prefix));
  }

  private final MetricRegistry metricRegistry;
  private final String prefix;

  @Override
  public void onGaugeAdded(String name, Gauge<?> gauge) {
    // entries in MetricRegistry must be unique, so replace any existing metric
    // (currently this happens for model batch mode, where we use several different
    // ExecutionContexts, each with their own MetricRegistry)
    metricRegistry.remove(prefix + name);
    metricRegistry.register(prefix + name, gauge);
  }

  @Override
  public void onGaugeRemoved(String name) {
    metricRegistry.remove(prefix + name);
  }

  @Override
  public void onCounterAdded(String name, Counter counter) {
    metricRegistry.remove(prefix + name);
    metricRegistry.register(prefix + name, counter);
  }

  @Override
  public void onCounterRemoved(String name) {
    metricRegistry.remove(prefix + name);
  }

  @Override
  public void onHistogramAdded(String name, Histogram histogram) {
    metricRegistry.remove(prefix + name);
    metricRegistry.register(prefix + name, histogram);
  }

  @Override
  public void onHistogramRemoved(String name) {
    metricRegistry.remove(prefix + name);
  }

  @Override
  public void onMeterAdded(String name, Meter meter) {
    metricRegistry.remove(prefix + name);
    metricRegistry.register(prefix + name, meter);
  }

  @Override
  public void onMeterRemoved(String name) {
    metricRegistry.remove(prefix + name);  }

  @Override
  public void onTimerAdded(String name, Timer timer) {
    metricRegistry.remove(prefix + name);
    metricRegistry.register(prefix + name, timer);
  }

  @Override
  public void onTimerRemoved(String name) {
    metricRegistry.remove(prefix + name);
  }

}
