/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.util;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.PrintStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.util.Comparator;
import java.util.function.Consumer;
import java.util.function.Predicate;

import nz.org.riskscape.engine.FileProblems;
import nz.org.riskscape.engine.RiskscapeIOException;
import nz.org.riskscape.problem.Problem;
import nz.org.riskscape.problem.Problems;

/**
 * Utility functions to help with file operations.
 */
public class FileUtils {

  /**
   * Rewrite the contents for the file specified. Writing is done to a .tmp file, so
   * we don't obliterate any old copy of the file until we have a good replacement.
   */
  public static void rewrite(Path outputFilepath, Consumer<PrintStream> writer) {
    File tmpFile = new File(outputFilepath.toString() + ".tmp");
    PrintStream out;
    try {
      out = new PrintStream(tmpFile);
    } catch (FileNotFoundException e) {
      throw new RiskscapeIOException(writeError(tmpFile.toPath(), e));
    }
    writer.accept(out);
    out.close();
    try {
      Files.move(tmpFile.toPath(), outputFilepath, StandardCopyOption.ATOMIC_MOVE, StandardCopyOption.REPLACE_EXISTING);
    } catch (IOException e) {
      throw new RiskscapeIOException(writeError(outputFilepath, e));
    }
  }

  /**
   * Removes the directory along with anything in it.
   * @param directory to remove
   * @throws IOException should anything go wrong
   */
  public static void removeDirectory(Path directory) throws IOException {
    cleanDirectory(directory, path -> true);
  }

  /**
   * Cleans a directory by removing any files/directories contained in it. The directory itself is
   * left alone.
   *
   * @param directory to clean
   * @throws IOException should anything go wrong
   */
  public static void cleanDirectory(Path directory) throws IOException {
    cleanDirectory(directory, path -> ! path.equals(directory));
  }

  private static void cleanDirectory(Path directory, Predicate<Path> filter) throws IOException {
    if (!Files.exists(directory)) {
      return;
    }
    if (!Files.isDirectory(directory)) {
      throw new IllegalArgumentException("Not a directory: " + directory.toString());
    }

    Files.walk(directory)
        .sorted(Comparator.reverseOrder())
        .filter(filter)
        .map(Path::toFile)
        .forEach(File::delete);
  }

  private static Problem writeError(Path filepath, Exception ex) {
    return Problems.get(FileProblems.class).cantWriteTo(filepath)
        .withChildren(Problems.caught(ex));
  }
}
