/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.util;

import java.io.PrintStream;
import java.nio.file.Path;
import java.util.function.Consumer;

import com.codahale.metrics.MetricRegistry;

import lombok.extern.slf4j.Slf4j;
import nz.org.riskscape.engine.OsUtils;
import nz.org.riskscape.engine.RiskscapeIOException;
import nz.org.riskscape.problem.Problem.Severity;
import nz.org.riskscape.problem.ProblemSink;
import nz.org.riskscape.problem.Problems;

/**
 * Periodically outputs a pipeline's progress to a file.
 */
@Slf4j
public class FileProgressNotifier extends ProgressNotifier {

  // update file every second. Anything quicker is probably unnecessary disk churn
  public static final long INITIAL_DELAY = 1000;
  public static final long UPDATE_INTERVAL = 1000;

  private final Path outputPath;
  private final ProblemSink problemSink;
  private boolean problemLogged = false;

  public FileProgressNotifier(MetricRegistry metrics, Path outputPath, ProblemSink sink) {
    this(metrics, outputPath, sink, INITIAL_DELAY, UPDATE_INTERVAL);
  }

  protected FileProgressNotifier(MetricRegistry metrics, Path outputPath, ProblemSink sink, long initialDelayMs,
      long updateIntervalMs) {
    super(metrics, initialDelayMs, updateIntervalMs);
    this.outputPath = outputPath;
    this.problemSink = sink;
  }

  @Override
  protected void updateStatus() {
    writeFile(out -> {
      out.println(String.join(OsUtils.LINE_SEPARATOR, buildUpdateMessage()));
    });
  }

  @Override
  public void finish() {
    super.finish();

    writeFile(out -> {
      // include final stats in output
      out.println(String.join(OsUtils.LINE_SEPARATOR, buildUpdateMessage()));
      out.println("Complete!");
    });
  }

  private void writeFile(Consumer<PrintStream> writer) {
    try {
      FileUtils.rewrite(outputPath, writer);
    } catch (RiskscapeIOException ex) {
      // if we failed to write the progress, then it's not the end of the world.
      // log it once and try again next time, in case it was a one-off problem
      if (!problemLogged) {
        problemSink.log(Problems.caught(ex).withSeverity(Severity.WARNING));
        problemLogged = true;
      }
    }
  }
}
