/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */

package nz.org.riskscape.engine.resource;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.Arrays;
import java.util.stream.Collectors;

import lombok.EqualsAndHashCode;
import nz.org.riskscape.config.ConfigString;
import nz.org.riskscape.engine.OsUtils;
import nz.org.riskscape.engine.RiskscapeIOException;

/**
 * A {@link Resource} that is backed by a {@link String}.
 */
@EqualsAndHashCode(callSuper = true)
public class StringResource extends BaseResource {

  private static final char ELLIPSIS = '\u2026';

  // the resource's content, e.g. the data, the value, the big kahuna
  private final String content;

  /**
   * Create a resource from a configuration defined value
   */
  public static StringResource fromConfig(ConfigString config) {
    return new StringResource(config.getLocation(), config.toString());
  }

  /**
   * Creates an resource for use internally within the engine, e.g. turning
   * wizard-generated code into a pipeline Resource. This helps to provide a
   * little more context in errors than Resource.UNKNOWN.
   *
   * @param fixedName       sensible text to refer to this resource. Must be valid
   *                        URI syntax, i.e. use dashes instead of spaces
   * @param content the actual text to wrap up in a Resource
   */
  public static StringResource internal(String fixedName, String content) {
    try {
      return new StringResource(fixedName, content);
    } catch (URISyntaxException ex) {
      // internal error - you should only be calling this with a fixed string you know to be valid
      throw new AssertionError(String.format("'%s' is not a valid URI: %s", fixedName, ex.toString()));
    }
  }

  public StringResource(URI location, String content) {
    super(location);
    this.content = content;
  }

  public StringResource(String location, String content) throws URISyntaxException {
    this(new URI(location), content);
  }

  public StringResource(String location, String... contentLines) throws URISyntaxException {
    this(
        location,
        Arrays.asList(contentLines).stream()
            .collect(Collectors.joining(OsUtils.LINE_SEPARATOR))
    );
  }

  @Override
  public InputStream getContentStream() throws RiskscapeIOException {
    return new ByteArrayInputStream(content.getBytes());
  }

  @Override
  public String toString() {
    String displayResource = content;

    if (displayResource.length() > 20) {
      // if resource is long we truncate it
      displayResource = displayResource.substring(0, 20) + ELLIPSIS;
    }

    return String.format("StringResource(%s)", displayResource);
  }
}
