/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.relation;

import java.util.ArrayList;
import java.util.List;

import com.google.common.collect.Lists;

import lombok.Getter;
import nz.org.riskscape.engine.types.Struct;

/**
 * Unifies a list of {@link Relation}s in to a single contiguous one.  Assumes that all given relations are of the same
 * type.
 */
public class UnionRelation extends BaseRelation {

  @Getter
  private final List<Relation> relations;

  public UnionRelation(Relation first, Relation... rest) {
    super(first.getType());

    this.relations = new ArrayList<>(rest.length + 1);
    relations.add(first);
    for (int i = 0; i < rest.length;) {
      relations.add(rest[i++]);
    }
  }

  public UnionRelation(Struct type, List<Relation> relations) {
    super(type);
    this.relations = relations;
  }

  public UnionRelation(Fields fields, List<Relation> relations) {
    super(fields);
    this.relations = relations;
  }

  @Override
  public String getSourceInformation() {
    return String.format("Union of %s... (size: %d)", relations.get(0), relations.size());
  }

  @Override
  protected TupleIterator rawIterator() {
    return new UnionIterator(Lists.transform(relations, src -> src.iterator()).iterator());
  }

  @Override
  protected BaseRelation clone(Fields fields) {
    return new UnionRelation(fields, relations);
  }


}
