/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.relation;

import java.util.Optional;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;

import com.codahale.metrics.Timer;

import nz.org.riskscape.engine.Tuple;

public class TupleIterators {
  /**
   * Returns a {@link TupleIterator} that uses a supplier function to return values for the tuple.  It is called
   * a peeker as it's often used in the cases where a straight wrappedAndMapped won't do, such as some filtering being
   * required and a peeking implementation is needed.
   * @param supplier a supplier of tuples to give to the {@link TupleIterator}.  Should return null to signal
   * that the supply is exhausted.
   */
  public static TupleIterator peeker(Supplier<Tuple> supplier, Optional<Runnable> onClose) {
    return new PeekingTupleIterator(onClose) {
      @Override
      protected Tuple get() {
        return supplier.get();
      }

      @Override
      protected String getSource() {
        return supplier.toString();
      }
    };
  }

  public static TupleIterator timed(Timer timer, TupleIterator iterator) {
    return timed(timer, iterator, null);
  }

  /**
   * @return a {@link TupleIterator} that wraps the given iterator to meter its output on calls to
   * {@link TupleIterator#next()}
   */
  public static TupleIterator timed(Timer timer, TupleIterator iterator, Runnable onClose) {
    return new TupleIterator() {

      @Override
      public boolean hasNext() {
        return iterator.hasNext();
      }

      @Override
      public Tuple next() {
        long nanos = System.nanoTime();
        try {
          return iterator.next();
        } finally {
          timer.update(System.nanoTime() - nanos, TimeUnit.NANOSECONDS);
        }
      }

      @Override
      public void close() {
        if (onClose != null) {
          onClose.run();
        }
        iterator.close();
      }
    };
  }
}
