/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.relation;

import java.util.NoSuchElementException;
import java.util.Optional;

import lombok.RequiredArgsConstructor;
import nz.org.riskscape.engine.Tuple;

/**
 * Base implementation of {@link TupleIterator} that needs to attempt to construct the next Tuple to know if there is
 * another one or not.
 */
@RequiredArgsConstructor
public abstract class PeekingTupleIterator implements TupleIterator {

  private final Optional<Runnable> onClose;

  private Tuple peeked;
  private InvalidTupleException caughtWhilePeeking;

  protected PeekingTupleIterator() {
    this(Optional.empty());
  }

  @Override
  public boolean hasNext() {
    return peek(false);
  }

  @Override
  public final Tuple next() {
    peek(true);
    Tuple toReturn = peeked;

    if (toReturn == null) {
      throw new NoSuchElementException();
    } else {
      peeked = null;
    }
    return toReturn;
  }

  @Override
  public void close() {
    onClose.ifPresent(Runnable::run);
  }

  private boolean peek(boolean throwPeeked) {

    if (throwPeeked && caughtWhilePeeking != null) {
      InvalidTupleException toThrow = caughtWhilePeeking;
      caughtWhilePeeking = null;
      throw toThrow;
    } else if (caughtWhilePeeking != null) {
      // a subsequent hasNext still needs to recognize the waiting time bomb
      return true;
    }

    if (peeked == null) {
      try {
        peeked = get();
      } catch (InvalidTupleException e) {
        if (throwPeeked) {
          throw e;
        } else {
          caughtWhilePeeking = e;
          return true;
        }
      }
    }
    return peeked != null;
  }

  /**
   * Attempt to construct the next tuple, returning null if the underlying source of data has been exhausted
   */
  protected abstract Tuple get();

  // TODO consider adding this to the TupleIterator interface
  protected abstract String getSource();

  @Override
  public String toString() {
    return String.format("Peeking(%s)", getSource());
  }

}
