/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.relation;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Optional;

import org.locationtech.jts.geom.Geometry;
import org.geotools.api.referencing.crs.CoordinateReferenceSystem;

import com.google.common.collect.ImmutableList;

import lombok.NonNull;
import nz.org.riskscape.engine.Project;
import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.query.TupleUtils;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Struct.StructMember;
import nz.org.riskscape.engine.types.Type;
import nz.org.riskscape.engine.types.Types;

/**
 * A relation backed by a {@link List}.  Useful for testing and such.
 */
public class ListRelation extends BaseRelation {

  /**
   * Create a {@link ListRelation} that yields values of the given type as single-valued {@link Struct} based tuples.
   * Useful for tests where you don't care much about the type
   */
  public static ListRelation ofValues(Object... rest) {
    Type type = Types.fromJavaTypeOptional(rest[0].getClass()).get();
    Struct struct = type.asStruct();
    List<Tuple> tuples = new ArrayList<>(rest.length);
    for (int i = 0; i < rest.length; i++) {
      Object object = rest[i];
      tuples.add(Tuple.ofValues(struct, type.coerce(object)));
    }

    return new ListRelation(tuples);
  }

  /**
   * Like {@link #ofValues(Object...)} but with a specific type - useful when the single member isn't named value.
   */
  public static ListRelation ofTypedValues(Struct struct, Object... rest) {
    List<Tuple> tuples = new ArrayList<>(rest.length);
    for (int i = 0; i < rest.length; i++) {
      Object object = rest[i];
      tuples.add(Tuple.ofValues(struct, object));
    }

    return new ListRelation(tuples);
  }

  @NonNull
  private final List<Tuple> source;

  public ListRelation(Struct type, List<Tuple> source, SpatialMetadata spatialMetadata) {
    super(type, null, spatialMetadata);
    this.source = source;
  }

  public ListRelation(Struct type, List<Tuple> values) {
    this(type, values, null);
  }

  public ListRelation(List<Tuple> values) {
    this(values.get(0).getStruct(), values, null);
  }

  public ListRelation(Tuple first, Tuple... rest) {
    this(first.getStruct(), ImmutableList.<Tuple>builder().add(first).addAll(Arrays.asList(rest)).build());
  }

  public ListRelation(Struct type, Tuple... values) {
    this(type, Arrays.asList(values), null);
  }

  protected ListRelation(BaseRelation.Fields fields, List<Tuple> source) {
    super(fields);
    this.source = source;
  }

  @Override
  public TupleIterator rawIterator() {
    return TupleIterator.wrapped(source.iterator(), Optional.empty());
  }

  public List<Tuple> getList() {
    return source;
  }

  @Override
  public String toString() {
    return String.format("List[size:%d]", source.size());
  }

  @Override
  public String getSourceInformation() {
    return toString();
  }

  @Override
  protected BaseRelation clone(Fields fields) {
    return new ListRelation(fields, source);
  }

  /**
   * Build {@link SpatialMetadata} by looking at the type and geometry against the project.
   */
  public ListRelation inferSpatialMetadata(Project project) {
    StructMember geomMember = TupleUtils.findRequiredGeometryMember(getType());
    CoordinateReferenceSystem crs;
    if (source.size() > 0) {
      crs = project.getSridSet().get((Geometry) source.get(0).fetch(geomMember));
    } else {
      crs = project.getDefaultCrs();
    }

    return withSpatialMetadata(new SpatialMetadata(crs, geomMember));
  }

  public ListRelation withSpatialMetadata(SpatialMetadata newSpatialMetadata) {
    return new ListRelation(new BaseRelation.Fields().tap(f -> f.spatialMetadata = newSpatialMetadata), source);
  }

  @Override
  public Optional<Long> size() {
    return Optional.of(Long.valueOf(source.size()));
  }
}
