/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.output;

import java.util.*;

import lombok.EqualsAndHashCode;
import lombok.Getter;
import lombok.RequiredArgsConstructor;
import nz.org.riskscape.engine.bind.BindingContext;
import nz.org.riskscape.engine.bind.JavaParameterSet;
import nz.org.riskscape.engine.bind.Parameter;
import nz.org.riskscape.engine.problem.GeneralProblems;
import nz.org.riskscape.engine.problem.ProblemPlaceholder;
import nz.org.riskscape.engine.types.Geom;
import nz.org.riskscape.engine.types.Nullable;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Type;
import nz.org.riskscape.problem.Problem;
import nz.org.riskscape.problem.Problems;
import nz.org.riskscape.problem.ResultOrProblems;

/**
 * Simple base class for {@link Format} implementations that covers some of the basic housekeeping
 */
@RequiredArgsConstructor
@EqualsAndHashCode(of = "id")
public abstract class BaseFormat implements Format {

  /**
   * Useful method for determining if there's at least a single geometry member we can use to support writing tuples of
   * this type to a spatial format
   */
  public static boolean containsGeometry(Struct struct) {
    for (Struct.StructMember member : struct.getMembers()) {
      Type memberType = Nullable.strip(member.getType());
      if (memberType.find(Geom.class).isPresent()) {
        return true;
      }
      boolean foundInNested = memberType.find(Struct.class)
          .map(BaseFormat::containsGeometry)
          .orElse(false);
      if (foundInNested) {
        return foundInNested;
      }
    }

    return false;
  }

  @Getter
  private final String id;

  @Getter
  private final String extension;

  @Getter
  private final String mediaType;

  @Getter
  private final EnumSet<Characteristics> characteristics;

  public BaseFormat(String id, String extension, String mediaType) {
    this(id, extension, mediaType, EnumSet.noneOf(Characteristics.class));
  }

  public BaseFormat(String id, String extension, String mediaType, Characteristics first, Characteristics... rest) {
    this(id, extension, mediaType, EnumSet.of(first, rest));
  }

  @Override
  public String toString() {
    return String.format("Format[id=%s, extension=%s, mediaType=%s]", id, extension, mediaType);
  }

  @Override
  public ResultOrProblems<? extends FormatOptions> buildOptions(
          Map<String, List<?>> paramMap, BindingContext context, Struct input) {
    @SuppressWarnings("unchecked")
    JavaParameterSet<? extends FormatOptions> parameterSet =
            JavaParameterSet.fromBindingClass(getWriterOptionsClass());

    ProblemPlaceholder givenOptions = ProblemPlaceholder.of(Format.FormatOptions.class, getId());
    ResultOrProblems<? extends FormatOptions> built = parameterSet.bindToObject(context, paramMap)
            .flatMap((bound, probs) -> {
              if (!bound.getExtraneous().isEmpty()) {
                // XXX go boom
                List<Problem> problems = new ArrayList<>();
                List<String> possible = parameterSet.getDeclared().stream().map(Parameter::getName).toList();
                // XXX use the soundex/did you mean code here
                bound.getExtraneous().keySet().stream()
                        .forEach(key -> problems.add(
                                GeneralProblems.get().notAnOption(key, givenOptions, possible)
                        ));
                return ResultOrProblems.failed(problems);
              }
              return ResultOrProblems.of(bound, probs);
            })
            .composeProblems(Problems.foundWith(givenOptions))
            .map(bound -> bound.getBoundToObject());

    return built;
  }

  @Override
  public List<String> getRequiredOptions(BindingContext context) {
    @SuppressWarnings("unchecked")
    JavaParameterSet<? extends Format.FormatOptions> parameterSet =
            JavaParameterSet.fromBindingClass(getWriterOptionsClass());

    return parameterSet.getDeclared().stream()
            .filter(p -> !p.isOptional() && p.getDefaultValues(context).isEmpty())
            .map(Parameter::getName)
            .toList();
  }
}
