/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.i18n;

import java.lang.reflect.Field;
import java.util.ResourceBundle;

/**
 * Utility class for obtaining a {@link nz.org.riskscape.engine.i18n.RiskscapeMessage}.
 */
public class RiskscapeMessageUtils {

  /**
   * {@link ResourceBundle} code expected to contain a layout message for displaying enum values.
   *
   * This message should expect to receive two arguments being:
   * {0}: value text
   * {1}: {@link nz.org.riskscape.engine.i18n.RiskscapeMessage} containing description of enum value, defaults
   * to empty string
   */
  public static final String ENUM_VALUE_CODE = "enum.value";

  /**
   * Returns a {@link nz.org.riskscape.engine.i18n.RiskscapeMessage} suitable labelling a fieldName.
   *
   * @param fieldName name of field to get label for
   * @param clazz class containing the field
   * @return message for field label
   */
  public static RiskscapeMessage forFieldLabel(String fieldName, Class<?> clazz) {
    //Naive initial implementation, need a key for this clazz.
    //TODO consider if we should support keys on superclasses/interfaces
    return RiskscapeMessage.withDefault(
        getCode(fieldName, clazz, ".label"),
        fieldName);
  }

  /**
   * Get a {@link nz.org.riskscape.engine.i18n.RiskscapeMessage} identifying fieldName.
   *
   * This code would normally contain a description of what fieldName is for.
   *
   * Message codes are of the form {package}.{clazz}.{fieldName}
   *
   * @param fieldName
   * @param clazz
   * @return message for fieldName
   */
  public static RiskscapeMessage forField(String fieldName, Class clazz) {
    return RiskscapeMessage.of(getCode(fieldName, clazz, ""));
  }

  /**
   * Get a {@link nz.org.riskscape.engine.i18n.RiskscapeMessage} for fieldName.
   *
   * This code would normally contain a description of what fieldName is for.
   *
   * Message codes are of the form {package}.{clazz}.{fieldName}
   *
   * @param fieldName
   * @param clazz
   * @param defaultMessage to return when message cannot be resolved
   * @return message for fieldName
   */
  public static RiskscapeMessage forFieldWithDefault(String fieldName, Class clazz, String defaultMessage) {
    return RiskscapeMessage.withDefault(getCode(fieldName, clazz, ""), defaultMessage);
  }

  /**
   * Get a {@link nz.org.riskscape.engine.i18n.RiskscapeMessage} with all the potential codes that may be used as I18N
   * translation code for field contained in clazz in priority order (highest to lowest).
   *
   * Codes are of the form: {package}.{clazz}.{fieldName}
   *
   * Additional codes are added by ascending the class structure (from clazz) up to the class that declares
   * field.
   *
   * If field is an {@link Enum} type then the return {@link nz.org.riskscape.engine.i18n.RiskscapeMessage} will
   * also contain child {@link nz.org.riskscape.engine.i18n.RiskscapeMessage}s for each enum value. This child
   * elements will have codes of the form:
   * {package}.{enum class}.{enum value}
   *
   * @param field to build keys for
   * @param clazz class that contains field
   * @param defaultMessage to return when message cannot be resolved
   * @return message for field
   */
  public static RiskscapeMessage forFieldWithDefault(Field field, Class clazz, String defaultMessage) {
    String code = getCode(field.getName(), clazz, "");
    return RiskscapeMessage.of(code, defaultMessage);
  }

  private static String getCode(String fieldName, Class clazz, String suffix) {
    return String.format("%s.%s%s", clazz.getCanonicalName(), fieldName, suffix);
  }

}
