/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.i18n;

import java.text.MessageFormat;
import java.util.Collections;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;

import com.google.common.collect.Maps;

/**
 * A simple {@link MessageSource} that allows messages to be registered programmatically for a locale, as well as a
 * fallback no-locale message.  Doesn't have the same locale precedence logic that a ResourceBundle backed MessageSource
 * would - the locale must match completely.
 */
public class MutableMessageSource implements MessageSource {

  private Map<String, Map<Locale, String>> localeSpecificStore = Maps.newHashMap();
  private Map<String, String> nonSpecificLocaleStore = Maps.newHashMap();

  public MutableMessageSource() {
  }

  @Override
  public String getMessage(MessageKey resolvable) {
    return getMessage(resolvable.getCode(), resolvable.getMessageArguments(), resolvable.getDefaultMessage(),
        Locale.getDefault());
  }

  @Override
  public String getMessage(MessageKey resolvable, Locale locale) {
    return getMessage(resolvable.getCode(), resolvable.getMessageArguments(), resolvable.getDefaultMessage(), locale);
  }

  @Override
  public String getMessage(String code, Object... args) {
    return getMessage(code, args, Locale.getDefault());
  }

  @Override
  public String getMessage(String code, Object[] args, Locale locale) {
    return getMessage(code, args, null, locale);
  }

  @Override
  public String getMessage(String code, Object[] args, String defaultMessage, Locale locale) {
    String message = localeSpecificStore.getOrDefault(code, Collections.emptyMap()).get(locale);

    if (message == null) {
      message = nonSpecificLocaleStore.get(code);
    }

    if (message == null) {
      return defaultMessage;
    } else {
      return new MessageFormat(message).format(args);
    }
  }

  public void addMessage(String code, String messageFormat) {
    nonSpecificLocaleStore.put(code, messageFormat);
  }

  public void addMessage(Locale locale, String code, String messageFormat) {
    localeSpecificStore.computeIfAbsent(code, k -> new HashMap<>()).put(locale, messageFormat);
  }

  @Override
  public <T extends MessageFactory> T getMessageFactory(Class<T> factory, Locale locale) {
    return MessageFactoryProxy.getMessageFactory(factory, locale, this);
  }

}
