/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */

package nz.org.riskscape.engine.i18n;

import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;
import java.util.Locale;

import com.google.common.base.CaseFormat;

import lombok.RequiredArgsConstructor;

/**
 * Acts as a Proxy for a given {@link MessageFactory} to provide {@link MessageKey} and {@link String} returning
 * implementations based on the called method's signature.
 */
@RequiredArgsConstructor
public class MessageFactoryProxy implements InvocationHandler {
  /**
   * Returns an implementation of the given factory class that resolves messages against the given locale and message
   * source.  Defers to {@link MessageFactoryProxy}.
   */
  @SuppressWarnings("unchecked")
  public static <T extends MessageFactory> T getMessageFactory(
      Class<T> factory,
      Locale locale,
      MessageSource messageSource
  ) {
    return (T) Proxy.newProxyInstance(
        factory.getClassLoader(),
        new Class[] {factory},
        new MessageFactoryProxy(locale, messageSource));
  }


  private static final Object[] NO_ARGS = new Object[0];
  private final Locale locale;
  private final MessageSource messageSource;

  @Override
  public Object invoke(Object o, Method method, Object[] args) throws Throwable {
    String messageCode = method.getDeclaringClass().getCanonicalName()
          + "."
          + CaseFormat.LOWER_CAMEL.to(CaseFormat.UPPER_UNDERSCORE, method.getName());
    if (args == null) {
      args = NO_ARGS;
    }

    String defaultMessage = methodNameToMessage(method.getName());

    if (method.getReturnType().equals(MessageKey.class)) {
      return RiskscapeMessage.withDefault(messageCode, defaultMessage, args);
    }

    if (method.getReturnType().equals(String.class)) {
      return messageSource.getMessage(messageCode, args, defaultMessage, locale);
    }

    throw new RuntimeException(method + " must return a String or a MessageKey");
  }

  private String methodNameToMessage(String methodName) {
    return CaseFormat.LOWER_CAMEL.converterTo(CaseFormat.LOWER_UNDERSCORE).convert(methodName).replace('_', ' ');
  }

}
