/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.data;

import java.util.List;
import java.util.Optional;

import org.geotools.api.coverage.grid.GridCoverage;

import lombok.Getter;
import lombok.NonNull;
import nz.org.riskscape.engine.relation.Relation;
import nz.org.riskscape.engine.types.Type;
import nz.org.riskscape.problem.Problem;
import nz.org.riskscape.problem.Problems;
import nz.org.riskscape.problem.ResultOrProblems;

/**
 * Implementation of a {@link ResolvedBookmark} that works closely with the {@link BaseBookmarkResolver} class
 * to build {@link Relation} and {@link GridCoverage} objects
 *
 * @param <T> probably one of {@link Relation} or {@link GridCoverage}
 */
public final class ResolvedBookmarkImpl<T extends BookmarkParameters> implements ResolvedBookmark {

  @NonNull
  protected final BaseBookmarkResolver<T> resolver;

  @Getter
  protected final T paramsObject;

  private ResultOrProblems<Object> built;

  public ResolvedBookmarkImpl(BaseBookmarkResolver<T> resolver, T paramsObject) {
    this.resolver = resolver;
    this.paramsObject = paramsObject;
  }

  @Override
  public Bookmark getBookmark() {
    return paramsObject.bookmark;
  }

  @Override
  public <X> Optional<ResultOrProblems<X>> getIfType(Class<X> requiredType) {
    if (requiredType.isAssignableFrom(paramsObject.getDataType())) {
      return Optional.of(build().map(b -> requiredType.cast(b)));
    } else {
      return Optional.empty();
    }
  }

  @SuppressWarnings("unchecked")
  protected ResultOrProblems<Object> build() {
    if (hasValidationErrors()) {
      return ResultOrProblems.failed(Problem.error(
          "There are validation problems with this bookmark preventing it from being opened"));
    }

    if (built == null) {
      try {
        built = resolver.build(paramsObject);
      } catch (RuntimeException e) {
        // as an adjunct to the fix for #1291, we also try to prevent unhandled exceptions in bookmarks from bringing
        // down a whole project
        return ResultOrProblems.failed(Problems.caught(e));
      }
    }

    return built;
  }

  @Override
  public Type getScalarType() {
    return paramsObject.getScalarType(this);
  }

  @Override
  public String toString() {
    return String.format("%s(format=%s, location=%s)",
        paramsObject.bookmark.getId(),
        resolver.getFormat(paramsObject.bookmark),
        paramsObject.bookmark.getLocation());
  }

  @Override
  public List<Problem> validate() {
    return paramsObject.problems;
  }

  @Override
  public Class<?> getDataType() {
    return paramsObject.getDataType();
  }

}
