/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.bind;

import nz.org.riskscape.problem.Problems;
import nz.org.riskscape.problem.ResultOrProblems;

/**
 * Base class for older {@link ParameterBinder} classes to implement to allow them to work in the new and the old way
 * (throwing and non-throwing) without having to be rewritten
 *
 * NB there are only a couple of uses of this left, both of which are required for the old-style default model.  If
 * we do drop the default model, or overhaul it, then we can drop the CompatBinder.
 */
public abstract class CompatBinder extends BaseBinder {

  @Override
  public boolean canBind(Class<?> sourceType, Class<?> destinationType) {
    return canBind(Parameter.required("dummy", destinationType));
  }

  @Override
  public <T> ResultOrProblems<T> bindValue(BindingContext context, Object value, Class<T> destinationType) {

    Parameter dummy = Parameter.required("dummy", destinationType);

    try {
      return ResultOrProblems.of(destinationType.cast(bind(context, dummy, value.toString())));
    } catch (ParameterBindingException e) {
      return ResultOrProblems.failed(Problems.caught(e));
    }
  }

  /**
   * The original ParameterBinder#canBind method, which is removed and now exists only in here
   */
  protected abstract boolean canBind(Parameter parameter);

  // marked as abstract again - we don't want base binder's default, each older-style binder will implement this
  // (other wise why use this class?)
  @Override
  public abstract Object bind(BindingContext context, Parameter modelParameter, String value)
      throws ParameterBindingException;
}
