/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.bind;

import java.util.LinkedList;
import java.util.List;

import com.google.common.base.CaseFormat;

import lombok.RequiredArgsConstructor;
import nz.org.riskscape.engine.FailedObjectException;
import nz.org.riskscape.engine.Identified;
import nz.org.riskscape.engine.IdentifiedCollection;
import nz.org.riskscape.engine.NoSuchObjectException;
import nz.org.riskscape.problem.Problem;
import nz.org.riskscape.problem.Problems;
import nz.org.riskscape.problem.ResultOrProblems;

/**
 * A very simple base implementation of {@link ParameterBinder} that assigns an id based on the class's name and
 * implements the now deprecated (but used everywhere) throwing version of bind.
 */
@RequiredArgsConstructor
public abstract class BaseBinder implements ParameterBinder {

  @Override
  public String getId() {
    return CaseFormat.UPPER_CAMEL.to(CaseFormat.LOWER_HYPHEN, getClass().getSimpleName());
  }

  @Override
  public Object bind(BindingContext context, Parameter modelParameter, String value) throws ParameterBindingException {
    return bindValue(context, value, modelParameter.getType())
        .orElseThrow(ps -> new ParameterBindingException(modelParameter, ps));
  }

  protected <T extends Identified> ResultOrProblems<T> wrapLookupNullOnMissing(
      BindingContext context,
      Class<T> identifiedClass,
      String id
  ) {
    return wrapLookupInternal(context, identifiedClass, id, true);
  }

  protected <T extends Identified> ResultOrProblems<T> wrapLookup(
      BindingContext context,
      Class<T> identifiedClass,
      String id
  ) {
    return wrapLookupInternal(context, identifiedClass, id, false);
  }

  private <T extends Identified> ResultOrProblems<T> wrapLookupInternal(
      BindingContext context,
      Class<T> identifiedClass,
      String id,
      boolean nullOnMissing
  ) {
    IdentifiedCollection<T> collection = context.getProject().getCollectionByClass(identifiedClass);
    try {
      List<Problem> problems = new LinkedList<>();
      T found = collection.get(id, p -> problems.add(p));
      return ResultOrProblems.of(found, problems);
    } catch (FailedObjectException ex) {
      return ResultOrProblems.failed(Problems.caught(ex));
    } catch (NoSuchObjectException ex) {
      if (nullOnMissing) {
        return null;
      } else {
        return ResultOrProblems.failed(ex.getProblem());
      }
    }
  }
}
