/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.cli.tests;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

import org.hamcrest.Description;
import org.hamcrest.Matcher;
import org.hamcrest.TypeSafeDiagnosingMatcher;

import nz.org.riskscape.engine.cli.Table;
import nz.org.riskscape.engine.util.Pair;
import nz.org.riskscape.picocli.CommandLine.Help.Ansi.Text;

/**
 * Match/assert table output with these
 */
@SuppressWarnings("unchecked")
public class TableMatchers {

  /**
   * Matches a table row with the given id and values
   * @param idMatch matches a row with an id column.  `Pair.of("foo", "bar")` would match the first row with 'bar' in
   * column foo
   * @param valueMatchers a list of matchers to apply to the various columns in the row.
   * `Pair.of("foo", containsString("bar"))` will match the text bar occurring in the foo column
   */
  public static Matcher<Table> hasRow(
      Pair<String, String> idMatch,
      Pair<String, Matcher<String>>... valueMatchers) {

    return new TypeSafeDiagnosingMatcher<Table>(Table.class) {

      @Override
      public void describeTo(Description description) {
        description.appendText("table with row identified by").appendValue(idMatch).appendText("and rows");
        for (Pair<String, Matcher<String>> matcher : valueMatchers) {
          description.appendValue(matcher.getLeft()).appendText(" matching ").appendDescriptionOf(matcher.getRight());
        }
      }

      @Override
      protected boolean matchesSafely(Table item, Description mismatchDescription) {
        List<Text> header = item.getHeader();

        List<String> ids = new ArrayList<>(item.getRows().size());

        List<List<Text>> rows = item.getRows();
        for (List<Text> row : rows) {
          List<Pair<String, String>> paired = new ArrayList<>(row.size());
          int i = 0;
          for (; i < row.size(); i++) {
            paired.add(Pair.of(header.get(i).plainString(), row.get(i).plainString()));
          }

          paired.forEach(pair -> {
            if (pair.getLeft().equals(idMatch.getLeft())) {
              ids.add(pair.getRight());
            }
          });

          if (paired.stream().anyMatch(p -> p.equals(idMatch))) {
            for (Pair<String, Matcher<String>> valMatcher : valueMatchers) {
              Optional<Pair<String, String>> found =
                  paired.stream().filter(pair -> pair.getLeft().equals(valMatcher.getLeft())).findFirst();

              if (found.isPresent()) {
                if (!valMatcher.getRight().matches(found.get().getRight())) {
                  mismatchDescription.appendText("column ").appendValue(found.get().getLeft())
                    .appendText(" did not match ").appendDescriptionOf(valMatcher.getRight())
                    .appendText(", ");

                  valMatcher.getRight().describeMismatch(found.get().getRight(), mismatchDescription);
                  return false;
                }
              } else {
                mismatchDescription.appendText("has no column named ").appendValue(valMatcher.getLeft());
                return false;
              }
            }
            return true;
          }
        }

        mismatchDescription
          .appendText("no rows from ").appendValue(ids)
          .appendText(" matched id ").appendValue(idMatch);

        return false;
      }
    };

  }

}
