/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.cli.model;

import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import java.net.URI;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.List;
import java.util.Map;

import org.junit.After;
import org.junit.Test;

import com.google.common.collect.Lists;

import nz.org.riskscape.engine.TemporaryDirectoryTestHelper;

public class RunCommandTest implements TemporaryDirectoryTestHelper {

  RunCommand subject = new RunCommand();
  Path tempDir = null;

  @After
  public void cleanup() throws Exception {
    if (tempDir != null) {
      remove(tempDir);
    }
  }

  @Test
  public void readsParametersFromParameterArgs() {
    subject.parameters = Lists.newArrayList(
        "p1=foo",
        "p1=bar",
        "p2=baz"
    );
    Map<String, List<String>> parameters = subject.getCliParameterMap();
    assertThat(parameters.size(), is(2));
    assertThat(parameters, hasEntry("p1", Lists.newArrayList("foo", "bar")));
    assertThat(parameters, hasEntry("p2", Lists.newArrayList("baz")));
  }

  @Test
  public void readsParametersFromIniFile() throws Exception {
    subject.parametersFile = writeFile("parameters.ini",
        "[parameters]",
        "p1 = foo",
        "p1 = bar");

    Map<String, List<String>> parameters = subject.getCliParameterMap();
    assertThat(parameters.size(), is(1));
    assertThat(parameters, hasEntry("p1", Lists.newArrayList("foo", "bar")));
  }

  @Test
  public void relativePathInArgumentResolvedAgainstCurrentDirectory() {
    subject.parameters = Lists.newArrayList(
        "p1=./my-file"
    );
    URI expected = Paths.get("").toUri().resolve("my-file");
    Map<String, List<String>> parameters = subject.getCliParameterMap();
    assertThat(parameters.size(), is(1));
    assertThat(parameters, hasEntry("p1", Lists.newArrayList(expected.toString())));
  }

  @Test
  public void relativePathInParameterFileResolvedAgainstParameterFile() {
    subject.parametersFile = writeFile("parameters.ini",
        "[parameters]",
        "p1 = ./my-file");
    URI expected = subject.parametersFile.toUri().resolve("my-file");
    Map<String, List<String>> parameters = subject.getCliParameterMap();
    assertThat(parameters.size(), is(1));
    assertThat(parameters, hasEntry("p1", Lists.newArrayList(expected.toString())));
  }

  @Test
  public void parameterArgumentsTrumpParameterFile() {
    subject.parameters = Lists.newArrayList("p1=foo");
    subject.parametersFile = writeFile("parameters.ini",
        "[parameters]",
        "p1 = baz",
        "p1 = bar",
        "p2 = from-file");

    Map<String, List<String>> parameters = subject.getCliParameterMap();
    assertThat(parameters.size(), is(2));
    assertThat(parameters, hasEntry("p1", Lists.newArrayList("foo")));
    assertThat(parameters, hasEntry("p2", Lists.newArrayList("from-file")));
  }

  @Test
  public void parameterArgumentsAreTrimmed() {
    // User should be able to copy/paste parameters from ini files to the command line, they will
    // need to add quotes but RiskScape should figure the rest out.
    subject.parameters = Lists.newArrayList("p1 = foo  ");

    Map<String, List<String>> parameters = subject.getCliParameterMap();
    assertThat(parameters.size(), is(1));
    assertThat(parameters, hasEntry("p1", Lists.newArrayList("foo")));
  }

  private Path writeFile(String name, String... lines) {
    try {
      if (tempDir == null) {
        tempDir = createTempDirectory("RunCommandTest");
      }
      Path path = tempDir.resolve(name);
      Files.write(path, Lists.newArrayList(lines));
      return path;
    } catch (Exception e) {
      throw new RuntimeException(e);
    }
  }

}
