/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.cli;

import static org.junit.Assert.*;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;

import org.junit.Test;
import org.mockito.Mockito;

import nz.org.riskscape.engine.Assert;

public class InterruptibleReadlineTest {

  BlockingQueue<Byte> byteQueue = new ArrayBlockingQueue<>(100);
  boolean eof = false;

  // An input stream implementation that supports mark and reads bytes from a blocking queue and has an implementation
  // of available that "works"
  InputStream fakeStdin = new InputStream() {

    // when true, we return bytes from the markBuffer instead of the byte queue
    boolean resetting = false;
    ArrayList<Byte> markBuffer;


    @Override
    public int available() throws IOException {
      return byteQueue.size();
    }

    @Override
    public synchronized void mark(int readlimit) {
      markBuffer = new ArrayList<>(readlimit);
    };

    @Override
    public boolean markSupported() {
      return true;
    };

    @Override
    public synchronized void reset() throws IOException {
      resetting = true;
    };

    @Override
    public int read() throws IOException {

      if (resetting) {
        if (markBuffer.isEmpty()) {
          resetting = false;
        } else {
          return markBuffer.remove(0);
        }
      }

      try {
        byte read = byteQueue.take();
        if (markBuffer != null) {
          markBuffer.add(read);
        }
        return read;
      } catch (InterruptedException e) {
        return -1;
      }
    }
  };

  @Test
  public void failsIfStdinMarkNotSupported() {
    InputStream in = Mockito.mock(InputStream.class);
    Assert.assertThrows(IllegalArgumentException.class, () -> InterruptibleReadline.readline(in));
  }

  @Test
  public void canReadAMixOfTerminatedLines() throws Exception {
    InputStream in = new ByteArrayInputStream((
        "great stuff!\n" +
        "yeah man\r" +
        "all good\r\n" +
        "nasty\n" +
        "").getBytes());

    assertEquals("great stuff!", InterruptibleReadline.readline(in));
    assertEquals("yeah man", InterruptibleReadline.readline(in));
    assertEquals("all good", InterruptibleReadline.readline(in));
    assertEquals("nasty", InterruptibleReadline.readline(in));
  }

  @Test(timeout=2000)
  public void aLineCanBeReadInChunks() throws Exception {
    Thread writer = new Thread(() -> {
      try {
        Thread.sleep(100);

        addBytes("yes ");

        Thread.sleep(100);

        addBytes("lawd!");

        Thread.sleep(100);
        addBytes("\n");

      } catch (InterruptedException e) {
        e.printStackTrace();
      }
    });

    writer.start();

    assertEquals("yes lawd!", InterruptibleReadline.readline(fakeStdin));
  }


  @Test(timeout=3000)
  public void aThreadCanBeInterruptedWhileReading() throws Exception {
    Thread testThread = Thread.currentThread();

    Thread writer = new Thread(() -> {
      try {

        Thread.sleep(100);

        addBytes("yes ");

        Thread.sleep(1000);

        // send interrupt to thread doing readline
        testThread.interrupt();

      } catch (InterruptedException e) {
        e.printStackTrace();
      }
    });

    writer.start();

    Assert.assertThrows(InterruptedException.class, () -> InterruptibleReadline.readline(fakeStdin));
  }

  private void addBytes(String string) {
    for (byte b : string.getBytes()) {
      byteQueue.add(b);
    }
  }

}
