/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.cli;

import static org.junit.Assert.*;
import static org.hamcrest.Matchers.*;

import org.junit.Before;
import org.junit.Test;

import nz.org.riskscape.engine.function.StringFunctions;
import nz.org.riskscape.engine.rl.DefaultOperators;
import nz.org.riskscape.engine.rl.MathsFunctions;
import nz.org.riskscape.engine.types.CoercionException;
import nz.org.riskscape.engine.types.TypeRegistry;
import nz.org.riskscape.problem.Problem;

public class ExpressionCommandTest extends ApplicationCommandTest {

  ExpressionCommand.Eval eval = new ExpressionCommand.Eval();
  ExpressionCommand.Inspect inspect = new ExpressionCommand.Inspect();

  @Override
  public void populateProject() {
    project.getFunctionSet().insertFirst(DefaultOperators.INSTANCE);
    project.getFunctionSet().addAll(StringFunctions.FUNCTIONS);
    project.getFunctionSet().addAll(MathsFunctions.FUNCTIONS);
    TypeRegistry.addDefaults(engine.getTypeRegistry());
  }

  @Before
  public void setup() {
    addCommandDependencies(eval);
    addCommandDependencies(inspect);

  }

  @Test
  public void evaluatePrintsOutResult() throws Exception {
    eval.expression = "1";
    assertEquals("1", eval.run());
  }

  @Test
  public void evaluatePrintsOutAnErrorOnFailure() throws Exception {
    eval.expression = "foo";
    assertTrue(eval.run() instanceof Problem);
  }

  @Test
  public void evalExceptionHandled() throws Exception {
    eval.expression = "float('not a number')";
    Object result = eval.run();
    assertTrue(result instanceof Problem);
    Problem p = (Problem) result;
    assertThat(render(p), containsString("Failed to evaluate `float('not a number')`"));
    assertTrue(p.getException() instanceof CoercionException);
  }

  @Test
  public void inspectPrintsOutADescriptionOfTheExpression() throws Exception {
    inspect.expression = "str({foo: [1, ' ', (true or (false and false))]})";
    // this tests merely asserts that it returns a string for printing without choking on any particular part of the
    // input - not bothering to exhaustively test the input and output
    assertTrue(inspect.run() instanceof String);
  }

  @Test
  public void inspectReturnsAnErrorOnFailure() throws Exception {
    inspect.expression = "cool_function()";
    assertTrue(inspect.run() instanceof Problem);
  }

}
