/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.cli;

import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import java.io.IOException;
import java.nio.file.Paths;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import org.junit.Before;
import org.junit.Test;

import lombok.RequiredArgsConstructor;
import nz.org.riskscape.engine.Assert;
import nz.org.riskscape.engine.DefaultEngine;
import nz.org.riskscape.engine.bind.BindingContext;
import nz.org.riskscape.engine.bind.Parameter;
import nz.org.riskscape.engine.bind.ParameterSet;
import nz.org.riskscape.engine.data.Bookmark;
import nz.org.riskscape.engine.data.BookmarkResolver;
import nz.org.riskscape.engine.data.ResolvedBookmark;
import nz.org.riskscape.engine.i18n.ResourceClassLoader;
import nz.org.riskscape.picocli.CommandLine.Help.Ansi.Text;

public class BookmarkFormatTest extends EngineCommandTest {

  FormatsCommand.FormatListCommand listCommand = new FormatsCommand.FormatListCommand();
  FormatsCommand.FormatInfo infoCommand = new FormatsCommand.FormatInfo();

  @RequiredArgsConstructor
  public static class TestResolver implements BookmarkResolver {

    private final String name;

    @Override
    public Optional<ResolvedBookmark> resolve(Bookmark bookmark, BindingContext context) throws IOException {
      return null;
    }

    @Override
    public Set<String> getFormats() {
      return Collections.singleton(name + " format");
    }

    @Override
    public List<String> getExtensions(String format) {
      HashMap<String, String> formatMap = new HashMap<>();
      formatMap.put("foo format", ".bar");
      formatMap.put("bar format", ".foo");
      return Arrays.asList(formatMap.get(name + " format"));
    }

    @Override
    public String getName() {
      return name + " resolver";
    }

    @Override
    public ParameterSet getParameterSet() {
     return ParameterSet.from(
       Parameter.optional("foo", String.class),
       Parameter.required("bar", String.class, "bar-default"),
       Parameter.required("baz", Integer.class, 1)
     );
    }
  }

  @Before
  public void setup() throws Exception {

    engine.add(new TestResolver("foo"));
    engine.add(new TestResolver("bar"));

    addEngineCommandDependencies(listCommand);
    addEngineCommandDependencies(infoCommand);

    messages.getClassLoader().append(BookmarkFormatTest.class,
        new ResourceClassLoader(
            "test_i18n/",
            Paths.get("src", "test", "resources").toUri().toURL()
        )
    );

  }

  @Test
  public void canListAllResolvers() throws Exception {
    Table result = (Table) listCommand.doCommand(engine);
    assertThat(result.getRows(), contains(
        contains("bar format", ".foo", "input data"),
        contains("foo format", ".bar", "input data")
    ));
  }

  @Test
  public void canListOptionsOnAParticularFormat() throws Exception {

    infoCommand.setName("foo format");
    Table result = (Table) infoCommand.doCommand(engine);
    assertThat(result.getRows(), contains(
        contains("bar", "Important (default: bar-default)"),
        contains("baz", "(default: 1)"),
        contains("foo", "This is a fine parameters")
    ));
  }

  @Test
  public void failsIfResolverDoesNotExist() throws Exception {
    infoCommand.setName("Great Resolvee");

    Assert.assertThrows(ExitException.class, () -> infoCommand.doCommand(engine));
  }

  @Test
  public void defaultEngineResolversAreListed() throws Exception {
    engine = new DefaultEngine();
    Set<String> resolverNames = engine.getBookmarkResolvers().getAll()
      .stream()
      .map(br -> br.getName())
      .collect(Collectors.toSet());


    Table table = (Table) listCommand.doCommand(engine);
    for (List<Text> row : table.getRows()) {
      assertThat(resolverNames, hasItem(row.get(0).plainString()));
    }
  }

}
