/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.cli;

import static org.junit.Assert.*;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.junit.Test;
import org.mockito.Mockito;

import nz.org.riskscape.engine.bind.BindingContext;
import nz.org.riskscape.engine.bind.BoundParameters;
import nz.org.riskscape.engine.bind.Parameter;
import nz.org.riskscape.engine.bind.ParameterSet;
import nz.org.riskscape.problem.Problem;
import nz.org.riskscape.rl.ExpressionParser;

/**
 * Test various static methods on ApplicationCommand
 */
public class ApplicationCommandHelpersTest {

  ParameterSet parameterSet = ParameterSet.from(
    Parameter.required("foo", Integer.class),
    Parameter.range("bar", Boolean.class, 0, 3),
    Parameter.optional("baz", String.class)
  );

  List<Problem> validationProblems = new ArrayList<>();
  Map<String, List<?>> unbound = new HashMap<>();;
  Map<String, List<?>> valueMap = new HashMap<>();
  BindingContext context = Mockito.mock(BindingContext.class);

  @Test
  public void paramStringConvertsAParameterSetInToAMultiLineString() throws Exception {
    valueMap.put("foo", Arrays.asList(1));
    valueMap.put("bar", Arrays.asList(true, false));
    valueMap.put("baz", Arrays.asList("yay"));

    // note the sorting by alpha
    assertFormattedAs(""
        + "bar = [true, false]"
        + "\n"
        + "baz = [yay]"
        + "\n"
        + "foo = [1]"
        + ""
    );
  }

  @Test
  public void paramStringExcludesParametersThatHaveNotBeenSetOrAreEmpty() throws Exception {
    valueMap.put("foo", Arrays.asList(1));
    valueMap.put("bar", Arrays.asList());
    // baz not there - poor baz

    assertFormattedAs("foo = [1]");
  }

  @Test
  public void paramStringPrefersToShowUnboundParametersOverBoundOnes() throws Exception {
    // this will show as empty, so users know that they should set it to something at run time.
    valueMap.put("foo", Arrays.asList());
    unbound.put("foo", Arrays.asList("the number one"));

    valueMap.put("bar", Arrays.asList(false));
    unbound.put("bar", Arrays.asList("yes", "NO!!!"));

    // this shouldn't show - no value was bound, so misleading to show the unbound value as though it is there
    valueMap.put("baz", Arrays.asList());
    unbound.put("baz", Arrays.asList("text"));

    assertFormattedAs(""
        + "bar = [yes, NO!!!]"
        + "\n"
        + "foo = []"
        + ""
    );
  }

  @Test
  public void paramStringFormatsAnExpressionEitherBoundOrUnboundWithToSource() throws Exception {
    // some extra whitespace to show it's the result of toSource, rather than printing the token's source
    unbound.put("foo", Arrays.asList(ExpressionParser.parseString("zero -   12   = chilly")));
    // no value means we show nothing, so put something there
    valueMap.put("foo", Arrays.asList("we bound to something different"));

    // the type is ignored - an expression anywhere is toSourced
    valueMap.put("bar", Arrays.asList(ExpressionParser.parseString("foo(bar)")));

    assertFormattedAs(""
        + "bar = [foo(bar)]"
        + "\n"
        + "foo = [zero - 12 = chilly]"
        + ""
    );
  }

  @Test
  public void paramStringLeftAlignsValues() throws Exception {
    parameterSet = ParameterSet.from(
        Parameter.required("short", String.class),
        Parameter.required("this-one-is-longer", String.class)
    );

    valueMap.put("short", Arrays.asList("yes"));
    valueMap.put("this-one-is-longer", Arrays.asList("no"));

    assertFormattedAs(""
        + "short              = [yes]"
        + "\n"
        + "this-one-is-longer = [no]"
        + ""
    );
  }

  @Test
  public void parametersWithOutValuesAreListed() throws Exception {
    parameterSet = ParameterSet.from(
        Parameter.required("short", String.class),
        Parameter.required("this-one-is-longer", String.class)
    );

    valueMap.put("short", Arrays.asList("yes"));

    assertFormattedAs(""
        + "short              = [yes]"
        + "\n"
        + "this-one-is-longer = []"
        + ""
    );
  }

  public void assertFormattedAs(String expected) {
    BoundParameters bound = new BoundParameters(parameterSet, context, valueMap, unbound, validationProblems);
    String actual = ApplicationCommand.paramString(bound);
    assertEquals(expected, actual);
  }

  // TODO we could adapt this code to return ansi text, which shows unbound values in red if there's a validation
  // problem associated with it
//  @Test
//  public void paramStringCanFormatInvalidParametersAsWell() throws Exception {
//
//  }
}
