/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.cli.pipeline;

import java.net.URI;
import java.nio.file.Paths;

import nz.org.riskscape.engine.Project;
import nz.org.riskscape.engine.bind.BindingContext;
import nz.org.riskscape.engine.cli.ExitException;
import nz.org.riskscape.engine.output.Format;
import nz.org.riskscape.picocli.CommandLine.Option;
import nz.org.riskscape.problem.Problems;

/**
 * Mixin of CLI switches for any command that wants to use the {@link CliPipelineRunner}
 */
public class CliPipelineRunnerOptions {

  @Option(names = {"-o", "--output"}, descriptionKey = "common.run.output")
  public String output;

  @Option(names = {"-r", "--replace"}, descriptionKey = "common.run.replace")
  public boolean replace;

  @Option(names = {"--checksum"}, descriptionKey = "common.run.checksum")
  public boolean checksum;

  @Option(names = { "--progress-indicator" }, descriptionKey = "common.run.progress-indicator")
  public String progressIndicator = CliPipelineRunner.MONITOR_PROGRESS_DEFAULT;

  @Option(names = { "--show-engine-stats" }, descriptionKey = "common.run.engine-stats")
  public boolean engineStats;

  @Option(names = {"-f", "--format"}, descriptionKey = "common.run.format")
  public String format;

  /**
   * Gets the `output` parameter as a URI using the given output base
   *
   * NB this method is intended for use with '--output' options associated with pipeline output
   *
   * @throws ExitException should any binding problems occur.
   */
  public URI getOutputBase(Project project) throws ExitException {
    BindingContext ctx = project.newBindingContext()
        // this ignores any project-defined output-base-location in favour of always generating urls that are relative
        // to the process working dir.
        .withNewRelativeTo(Paths.get("").toUri());

    return ctx.bind(output, URI.class)
        .orElseThrow(problems -> new ExitException(Problems.foundWith("--output", problems)));
  }

  public Format getFormat(Project project) {
    if (format != null) {
      return project.newBindingContext().bind(format, Format.class).orElseThrow(problems -> {
        throw new ExitException(Problems.foundWith("--format", problems));
      });
    } else {
      return null;
    }
  }
}
