/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */

package nz.org.riskscape.engine.cli.model;

import java.util.Arrays;
import java.util.Formatter;

import nz.org.riskscape.engine.Project;
import nz.org.riskscape.engine.RiskscapeException;
import nz.org.riskscape.engine.cli.ApplicationCommand;
import nz.org.riskscape.engine.cli.ExitException;
import nz.org.riskscape.engine.cli.pipeline.CliPipelineRunner;
import nz.org.riskscape.engine.model.IdentifiedModel;
import nz.org.riskscape.engine.model.Model;
import nz.org.riskscape.engine.pipeline.ExecutionContext;
import nz.org.riskscape.engine.pipeline.PipelineExecutor;
import nz.org.riskscape.engine.pipeline.RealizedPipeline;
import nz.org.riskscape.picocli.CommandLine.Command;
import nz.org.riskscape.picocli.CommandLine.Option;
import nz.org.riskscape.problem.ProblemSink;
import nz.org.riskscape.problem.Problems;

@Command(name="run")
public class RunCommand extends BaseModelRunCommand {

  @Option(names = "--print")
  public Boolean printPipeline = false;

  @Option(names = "--graph")
  public Boolean graphPipeline = false;

  @Override
  public Object doCommand(Project useProject) {


    boolean parametersNeedUpdating = !(parameters.isEmpty() && parametersFile == null);

    IdentifiedModel identifiedModel = ApplicationCommand.getObject(
        useProject.getIdentifiedModels(),
        modelId,
        // NB don't report errors on first fetch if we're only going to go and rebuild it with new parameters
        parametersNeedUpdating ? ProblemSink.DEVNULL : getTerminal()
    );
    Model model = identifiedModel.getModel();

    if (parametersNeedUpdating) {
      model = updateParameters(model, getCliParameterMap());
    }

    // TODO check for surplus CLI parameters that were ignored and warn/error

    PipelineExecutor executor = useProject.getEngine().getPipelineExecutor();
    try (ExecutionContext executionContext = executor.newExecutionContext(useProject)) {
      if (printPipeline) {
        // try to print the pipeline even if it has errors - this is a special case
        // where --print could help diagnose the problem
        printPipeline(realize(model, executionContext, true), new Formatter(stdout()), this);
        return null;
      }

      RealizedPipeline realizedPipeline = realize(model, executionContext);
      // replace the name with the model's name - for reasons, the ModelFramework API doesn't know the models id/name
      // when it's being built, so we sub it in here
      realizedPipeline = realizedPipeline.withMetadata(realizedPipeline.getMetadata().withName(modelId));

      if (graphPipeline) {
        graph(realizedPipeline, stdout(), "digraph", modelId, Arrays.asList("compound=true"));
        return null;
      }

      CliPipelineRunner runner = new CliPipelineRunner(getTerminal());

      warnIfCannotOverrideFormat(realizedPipeline, useProject.getEngine().getProblemSink());

      try {
        runner.run(realizedPipeline, executor, useProject, runnerOptions);
      } catch (RiskscapeException ex) {
        throw new ExitException(Problems.foundWith(model, Problems.caught(ex)), ex);
      }

      return null;
    }
  }

}
