/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.cli;

import java.io.File;
import java.net.URI;
import java.time.LocalDateTime;
import java.util.Arrays;
import java.util.Formatter;
import java.util.List;

import nz.org.riskscape.dsl.LexerException;
import nz.org.riskscape.dsl.ParseException;
import nz.org.riskscape.engine.Engine;
import nz.org.riskscape.engine.OsUtils;
import nz.org.riskscape.engine.Project;
import nz.org.riskscape.engine.RiskscapeException;
import nz.org.riskscape.engine.bind.Parameter;
import nz.org.riskscape.engine.cli.Table.Property;
import nz.org.riskscape.engine.cli.pipeline.CliPipelineRunner;
import nz.org.riskscape.engine.cli.pipeline.CliPipelineRunnerOptions;
import nz.org.riskscape.engine.pipeline.ExecutionContext;
import nz.org.riskscape.engine.pipeline.PipelineExecutor;
import nz.org.riskscape.engine.pipeline.PipelineSteps;
import nz.org.riskscape.engine.pipeline.RealizedPipeline;
import nz.org.riskscape.engine.pipeline.Step;
import nz.org.riskscape.engine.resource.FileResourceLoader;
import nz.org.riskscape.engine.resource.Resource;
import nz.org.riskscape.engine.resource.ResourceLoadingException;
import nz.org.riskscape.engine.resource.StreamResource;
import nz.org.riskscape.engine.resource.StringResource;
import nz.org.riskscape.picocli.CommandLine.Command;
import nz.org.riskscape.picocli.CommandLine.Mixin;
import nz.org.riskscape.picocli.CommandLine.Option;
import nz.org.riskscape.picocli.CommandLine.Parameters;
import nz.org.riskscape.pipeline.PipelineMetadata;
import nz.org.riskscape.pipeline.PipelineParser;
import nz.org.riskscape.pipeline.ast.PipelineDeclaration;
import nz.org.riskscape.problem.Problem;
import nz.org.riskscape.problem.Problems;

@Command(name = "pipeline",
subcommands = {
  PipelineCommand.StepCmd.class,
  PipelineCommand.Eval.class
})
public class PipelineCommand extends StubCommand {

  @Command(name = "step",
  subcommands = {
    PipelineCommand.StepList.class,
    PipelineCommand.StepInfo.class
  })
  public static class StepCmd extends StubCommand {

  }

  @Command(name = "list")
  public static class StepList extends EngineOnlyCommand {

    @Override
    public Object doCommand(Engine useEngine) {
      PipelineSteps steps = useEngine.getPipelineSteps();

      return Table.fromList(steps.getAll(), Step.class, getMessages().getLabels(),
          Arrays.asList(
              Property.of("id", Step::getId),
              Property.of("inputNames", (s) -> {
                if (s.getInputNames().isEmpty()) {
                  //We don't want `[]` shown in the case of no named inputs
                  return "";
                }
                return s.getInputNames().toString();
              }),

              Property.of("description", s -> getMessages().getHelpMessage(s, "description").orElse(null))
          )
      );
    }
  }

  @Command(
      name = "info")
  public static class StepInfo extends EngineOnlyCommand {

    @Parameters(arity = "1", index = "0")
    public String stepId;

    @Override
    public Object doCommand(Engine useEngine) {
      Step step = ApplicationCommand.getObject(useEngine.getPipelineSteps(), stepId, getTerminal());

      List<Parameter> parameters = step.getParameterSet().toList();

      return Table.fromList(parameters,
          Parameter.class,
          getMessages().getLabels(),
          Arrays.asList(
              Property.of("name", Parameter::getName),
              Property.of("typeName", Parameter::getTypeName),
              Property.of("arity", Parameter::getArity),

              Property.of("description", p -> getMessages().getHelpMessage(step, "params", p.getName()).orElse(null))
          )
      );
    }
  }

  @Command(
      name = "eval",
      aliases = "evaluate")
  public static class Eval extends AbstractPipelineRunner {

    @Parameters(index = "0")
    public String pipelineFile;

    @Override
    public Object doCommand(Project useProject) {
      PipelineDeclaration ast = parsePipeline();
      return run(useProject, ast);
    }

    private PipelineDeclaration parsePipeline() throws RiskscapeException {
      Resource resource = getResourceFrom(pipelineFile);

      String pipelineSource = resource.getContentAsString();

      try {
        return PipelineParser.INSTANCE.parsePipeline(pipelineSource)
            .withMetadata(new PipelineMetadata(resource.getLocation(), "pipeline-eval", ""));
      } catch (ParseException | LexerException e) {
        throw new ExitException(
            appendFileTip(Problem.error("Failed to parse pipeline from " + resource.getLocation())
                .withChildren(e.getProblem())
            )
        );
      }
    }

    private Problem appendFileTip(Problem problem) {
      String examplePath;
      if (OsUtils.isWindows()) {
        examplePath = ".\\pipeline.txt, C:\\Users\\Ronnie\\Projects\\pipeline.txt";
      } else {
        examplePath = "./pipeline.txt, /home/ronnie/riskscape/pipeline.txt";
      }

      return problem.withChildren(
              Problem.info("To evaluate from a file, prefix the filename e.g. " + examplePath));
    }

    public boolean isThisDefinitelyAFile(String pipelineArg, File file) {
      if (file.exists() || pipelineArg.startsWith("/") || pipelineArg.startsWith("\\")
          ||pipelineArg.startsWith("./") || pipelineArg.startsWith(".\\")) {
        return true;
      }
      return false;
    }

    public Resource getResourceFrom(String pipelineArg) {
      File file = new File(pipelineArg.trim());
      // read from STDIN
      if (file.getName().equals("-")) {
        return new StreamResource(Resource.UNKNOWN_URI, getTerminal().getIn()) {
          @Override
          public String toString() {
            return "<STDIN>";
          }

          @Override
          public URI getLocation() {
            return URI.create("STDIN");
          }
        };
        // Check if pipelineArg is likely to be a file
      } else if (isThisDefinitelyAFile(pipelineArg, file)) {
        try {
          return new FileResourceLoader().load(file.toURI());
        } catch (ResourceLoadingException e) {
          throw new ExitException(
            1,
            Problem.warning("Failed to load pipeline from file").withChildren(Problems.caught(e))
          );
        }
      } else {
        return new StringResource(URI.create("command-line-argument"), pipelineArg);
      }
    }
  }

  abstract static class AbstractPipelineRunner extends ApplicationCommand implements PipelineRenderer {

    @Option(names = "--print")
    public Boolean printPipeline = false;

    @Mixin
    public CliPipelineRunnerOptions runnerOptions = new CliPipelineRunnerOptions();

    protected Object run(Project useProject, PipelineDeclaration ast) {

      PipelineExecutor executor = useProject.getEngine().getPipelineExecutor();

      try (ExecutionContext executionContext = executor.newExecutionContext(useProject)) {
        RealizedPipeline realized = executionContext.realize(ast);

        if (printPipeline) {
          printPipeline(realized, new Formatter(stdout()), this);
          return null;
        } else {
          if (realized.hasFailures()) {
            return Problem.composite(realized.getFailures(), "There were pipeline failures");
          } else {
            execute(realized.drainWarnings(useProject.getProblemSink()), executor, useProject);
            return null;
          }
        }
      } catch (RiskscapeException ex) {
        throw new ExitException(Problems.foundWith(ast, Problems.caught(ex)), ex);
      }
    }

    private void execute(RealizedPipeline realized, PipelineExecutor executor, Project useProject) {

      CliPipelineRunner runner = new CliPipelineRunner(getTerminal());
      runner.run(realized, executor, useProject, runnerOptions);
    }

    protected LocalDateTime getCurrentTime() {
      return LocalDateTime.now();
    }
  }
}
