/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.rl.ast;

import static org.hamcrest.CoreMatchers.*;
import static org.junit.Assert.*;

import org.junit.Test;

import nz.org.riskscape.dsl.Lexer;
import nz.org.riskscape.dsl.Token;
import nz.org.riskscape.rl.ExpressionParser;
import nz.org.riskscape.rl.TokenTypes;

public class ExpressionsTest {

  ExpressionParser parser = new ExpressionParser();

  @Test
  public void canSumExpressions() {
    Expression threePlusFour = Expressions.sum(parser.parse("3"), parser.parse("4"));
    assertThat(threePlusFour.toSource(), is("3 + 4"));

    assertThat(Expressions.sum(threePlusFour, parser.parse("5")).toSource(),
        is("3 + 4 + 5"));
  }

  @Test
  public void canQuoteTextCorrectly() throws Exception {
    assertText("hi");
    // double quotes are fine

    // single quotes must be escaped
    assertText("'hi'");

    // backslashes themselves are fine
    assertText("\\hi'");
  }

  @Test
  public void canQuoteIdentsCorrectly() throws Exception {
    // these can all be used verbatim
    assertIdent("foo");
    assertIdent("fooBar");
    assertIdent("foo_1");
    assertIdent("foo_1_");

    // spaces need quoting
    assertQIdent("foo bar");
    // double quotes also need quoting
    assertQIdent("\" Foo my Bar \"");

    // keywords are also quoted
    assertQIdent("null");
    // case insensitive
    assertQIdent("NUll");
    assertQIdent("true");
    assertQIdent("false");
    assertQIdent("and");
    assertQIdent("or");
    assertQIdent("as");
  }

  private void assertIdent(String toQuote) {
    String quoted = TokenTypes.quoteIdent(toQuote);

    Lexer<TokenTypes> lexer = new Lexer<>(TokenTypes.tokens(), quoted);
    Token token = lexer.next();
    assertEquals(TokenTypes.IDENTIFIER, token.type);
    assertEquals(toQuote, token.getValue());

    Token asToken = TokenTypes.identToken(toQuote);
    assertEquals(asToken.source, toQuote);
    assertEquals(asToken.value, toQuote);
    assertEquals(TokenTypes.IDENTIFIER, asToken.type);
  }


  private void assertQIdent(String toQuote) {
    String quoted = TokenTypes.quoteIdent(toQuote);

    Lexer<TokenTypes> lexer = new Lexer<>(TokenTypes.tokens(), quoted);
    Token token = lexer.next();
    assertEquals(TokenTypes.QUOTED_IDENTIFIER, token.type);
    assertEquals(toQuote, token.getValue());

    Token asToken = TokenTypes.identToken(toQuote);
    assertNotEquals(asToken.source, toQuote);
    assertEquals(asToken.value, toQuote);
    assertEquals(TokenTypes.QUOTED_IDENTIFIER, asToken.type);
  }

  void assertText(String toQuote) {
    String quoted = TokenTypes.quoteText(toQuote);
    Lexer<TokenTypes> lexer = new Lexer<>(TokenTypes.tokens(), quoted);
    Token token = lexer.next();
    assertEquals(TokenTypes.STRING, token.type);
    assertEquals(toQuote, token.getValue());
  }

}
