/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.rl.ast;

import static org.junit.Assert.*;

import org.junit.Test;

import nz.org.riskscape.rl.ExpressionParser;

@SuppressWarnings("unchecked")
public class ExpressionConverterTest {

  public static class TestExpressionConverter extends ExpressionConverter<Void> {

  }

  public ExpressionConverter subject = new TestExpressionConverter();
  public ExpressionParser expressionParser = new ExpressionParser();

  @Test
  public void constantReturnsSameExpression() {
    Expression expr = expressionParser.parse("'constant text'");

    assertSame(expr, expr.accept(subject, null));

    expr = expressionParser.parse("10");

    assertSame(expr, expr.accept(subject, null));
  }

  @Test
  public void binaryReturnsNewButEquivalentExpression() {
    Expression expr = expressionParser.parse("3 > 5");

    Expression converted = expr.<Void, Expression>accept(subject, null);
    assertNotSame(expr, converted);
    assertEquals(expr, converted);
  }

  @Test
  public void bracketedReturnsNewButEquivalentExpression() {
    Expression expr = expressionParser.parse("(3)");

    Expression converted = expr.<Void, Expression>accept(subject, null);
    assertNotSame(expr, converted);
    assertEquals(expr, converted);
  }

  @Test
  public void propertyAccessASameExpression() {
    Expression expr = expressionParser.parse("foo.bar");

    assertEquals(expr, expr.accept(subject, null));
  }

  @Test
  public void propertyAccessCopiesReceiverAsWell() {
    Expression expr = expressionParser.parse("foo().bar");
    Expression cloned = expr.<Void, Expression>accept(subject, null);
    assertEquals(expr, cloned);
    assertNotSame(expr, cloned);
    assertEquals(
      expr.isA(PropertyAccess.class).get().getReceiver(),
      cloned.isA(PropertyAccess.class).get().getReceiver()
    );
    assertNotSame(
      expr.isA(PropertyAccess.class).get().getReceiver(),
      cloned.isA(PropertyAccess.class).get().getReceiver()
    );
  }

  @Test
  public void functionCallReturnsNewButEquivalentExpression() {
    Expression expr = expressionParser.parse("foo(bar)");

    Expression converted = expr.<Void, Expression>accept(subject, null);
    assertNotSame(expr, converted);
    assertEquals(expr.toSource(), converted.toSource());
  }

  @Test
  public void listExpressionReturnsNewButEquivalentExpression() {
    Expression expr = expressionParser.parse("[foo, bar]");

    Expression converted = expr.<Void, Expression>accept(subject, null);
    assertNotSame(expr, converted);
    assertEquals(expr.toSource(), converted.toSource());
  }

  @Test
  public void structDeclarationReturnsNewButEquivalentExpression() {
    Expression expr = expressionParser.parse("{foo, bar}");

    Expression converted = expr.<Void, Expression>accept(subject, null);
    assertNotSame(expr, converted);
    assertEquals(expr.toSource(), converted.toSource());

    expr = expressionParser.parse("{foo as f1, bar as b1}");
    converted = expr.<Void, Expression>accept(subject, null);
    assertNotSame(expr, converted);
    assertEquals(expr.toSource(), converted.toSource());

    expr = expressionParser.parse("{f1: foo, b1: bar}");
    converted = expr.<Void, Expression>accept(subject, null);
    assertNotSame(expr, converted);
    assertEquals(expr.toSource(), converted.toSource());
  }
}
