/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.pipeline.ast;


import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import java.util.ArrayList;
import java.util.List;

import org.junit.Test;

import com.google.common.collect.ImmutableMap;

import nz.org.riskscape.rl.ExpressionParser;
import nz.org.riskscape.rl.ast.Constant;
import nz.org.riskscape.rl.ast.Expression;
import nz.org.riskscape.rl.ast.ParameterToken;

public class ParameterReplacerTest {

  ParameterReplacer replacer = new ParameterReplacer(ImmutableMap.of(
      parse("$foo", ParameterToken.class), parse("'foo-value'", Constant.class),
      parse("$bar", ParameterToken.class), parse("'bar-value'", Constant.class)
  ));

  List<ParameterToken> missing = new ArrayList<>();
  Expression withParameters;
  Expression replaced;

  @Test
  public void canReplaceValues() {
    withParameters = parse("str_length($foo) < str_length($bar)");
    replace();

    assertThat(replaced, is(parse("str_length('foo-value') < str_length('bar-value')")));
    assertThat(missing, empty());
  }

  @Test
  public void canReplaceValuesEveryWhereTheyOccur() {
    withParameters = parse("str_length($foo) < str_length($foo)");
    replace();

    assertThat(replaced, is(parse("str_length('foo-value') < str_length('foo-value')")));
    assertThat(missing, empty());
  }

  @Test
  public void spotsParametersWithNoReplacements() {
    withParameters = parse("str_length($foo) < str_length($baz)");
    replace();

    assertThat(replaced, is(parse("str_length('foo-value') < str_length($baz)")));
    assertThat(missing, contains(parse("$baz", ParameterToken.class)));
  }

  private void replace() {
    replaced = withParameters.accept(replacer, missing);
  }

  private Expression parse(String expressionString) {
    return parse(expressionString, Expression.class);
  }

  private <T extends Expression> T parse(String expressionString, Class<T> expected) {
    return expected.cast(ExpressionParser.INSTANCE.parseAllowParameters(expressionString));
  }

}
