/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.types.eqrule;

import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import org.junit.Test;
import org.locationtech.jts.geom.GeometryFactory;
import org.locationtech.jts.geom.LineString;
import org.locationtech.jts.geom.MultiLineString;
import org.locationtech.jts.geom.MultiPoint;
import org.locationtech.jts.geom.MultiPolygon;
import org.locationtech.jts.geom.Point;
import org.locationtech.jts.geom.Polygon;
import org.locationtech.jts.geom.PrecisionModel;

import nz.org.riskscape.engine.types.MultiGeom;
import nz.org.riskscape.engine.types.TypeRegistry;
import nz.org.riskscape.engine.types.TypeRulesTest;
import nz.org.riskscape.engine.types.Types;
import nz.org.riskscape.engine.typeset.TypeSet;

@SuppressWarnings("unchecked")
public class SingleGeomToMultiGeomRuleTest extends TypeRulesTest {

  private static final int SRID = 10;
  private final PrecisionModel precisionModel = new PrecisionModel();
  private final GeometryFactory gf = new GeometryFactory(precisionModel, SRID);

  @Override
  protected void addTypes() {
    registry = TypeRegistry.withDefaults();
    typeSet = new TypeSet(registry);
  }

  @Test
  public void canCoerceLineToMulti() {
    Coercer coercer = findEquivalenceCoerce(Types.LINE, MultiGeom.MULTI_LINE);
    assertNotNull(coercer);

    LineString input = gf.createLineString();

    MultiLineString mp = (MultiLineString)coercer.apply(input);
    assertThat(mp.getSRID(), is(SRID));
    assertThat(mp.getNumGeometries(), is(1));
    assertThat(mp.getGeometryN(0), is(input));
  }

  @Test
  public void canCoercePointToMulti() {
    Coercer coercer = findEquivalenceCoerce(Types.POINT, MultiGeom.MULTI_POINT);
    assertNotNull(coercer);

    Point input = gf.createPoint();

    MultiPoint mp = (MultiPoint)coercer.apply(input);
    assertThat(mp.getSRID(), is(SRID));
    assertThat(mp.getNumGeometries(), is(1));
    assertThat(mp.getGeometryN(0), is(input));
  }

  @Test
  public void canCoercePolygonToMulti() {
    Coercer coercer = findEquivalenceCoerce(Types.POLYGON, MultiGeom.MULTI_POLYGON);
    assertNotNull(coercer);

    Polygon input = gf.createPolygon();

    MultiPolygon mp = (MultiPolygon)coercer.apply(input);
    assertThat(mp.getSRID(), is(SRID));
    assertThat(mp.getNumGeometries(), is(1));
    assertThat(mp.getGeometryN(0), is(input));
  }

  @Test
  public void cannotCoerceMultiToSingle() {
    assertNull(findEquivalenceCoerce(MultiGeom.MULTI_POLYGON, Types.POLYGON));
    assertNull(findEquivalenceCoerce(MultiGeom.MULTI_POINT, Types.POINT));
    assertNull(findEquivalenceCoerce(MultiGeom.MULTI_LINE, Types.LINE));
  }

  @Test
  public void cannotCoerceUnrelatedShapesToMulti() {
    assertNull(findEquivalenceCoerce(Types.POLYGON, MultiGeom.MULTI_POINT));
    assertNull(findEquivalenceCoerce(Types.POLYGON, MultiGeom.MULTI_LINE));
    assertNull(findEquivalenceCoerce(Types.LINE, MultiGeom.MULTI_POINT));
    assertNull(findEquivalenceCoerce(Types.LINE, MultiGeom.MULTI_POLYGON));
    assertNull(findEquivalenceCoerce(Types.POINT, MultiGeom.MULTI_POLYGON));
    assertNull(findEquivalenceCoerce(Types.POINT, MultiGeom.MULTI_LINE));
  }

}
