/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.types.ancestor;

import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import java.util.Arrays;
import java.util.List;
import java.util.function.Function;

import org.geotools.api.referencing.crs.CoordinateReferenceSystem;
import org.geotools.referencing.crs.DefaultGeographicCRS;
import org.junit.Test;

import nz.org.riskscape.engine.types.Nullable;
import nz.org.riskscape.engine.types.RSList;
import nz.org.riskscape.engine.types.Referenced;
import nz.org.riskscape.engine.types.Type;
import nz.org.riskscape.engine.types.Types;
import nz.org.riskscape.engine.types.WithinSet;

public class SimpleRulesTest extends AncestorRuleTest {

  @Test
  public void ancestorOfTwoDifferentSimpleTypesIsNothing() throws Exception {
    assertTrue(typeSet.computeAncestorType(Types.TEXT, Types.FLOATING).isEmpty());
  }

  @Test
  public void ancestorOfTwoSimpleTypesIsThemselves() {
    assertUnconvertedAncestor(Types.TEXT, Types.TEXT, Types.TEXT);
  }

  @Test
  public void ancestorOfTwoWrappedSimpleTypesIsTheSimpleType() {
    assertUnconvertedAncestor(
      Types.TEXT,
      new WithinSet(Types.TEXT, "foo", "bar"),
      new WithinSet(Types.TEXT, "foo", "baz")
    );
  }


  @Test
  public void ancestorOfTwoSimilarlyWrappedSimpleTypesIsTheWrappingType() {
    assertUnconvertedAncestor(
      new WithinSet(Types.TEXT, "foo", "bar"),
      new WithinSet(Types.TEXT, "foo", "bar"),
      new WithinSet(Types.TEXT, "foo", "bar")
    );

    // and with three layers of wrapping...
    assertUnconvertedAncestor(
      new WithinSet(Types.TEXT, "foo", "bar"),
      new WithinSet(Types.TEXT, "foo", "bar"),
      typeSet.add("id", new WithinSet(Types.TEXT, "foo", "bar"))
    );
  }

  @Test
  public void ancestorOfSimpleTypeAndWrappedIsTheSimpleType() {
    assertUnconvertedAncestor(
      Types.TEXT,
      new WithinSet(Types.TEXT, "foo", "bar"),
      Types.TEXT
    );
  }

  @Test
  public void ancestorOfTwoReferencedTypesIsReferenced() {
    // see ReferencedTest for more test cases
    CoordinateReferenceSystem crs = DefaultGeographicCRS.WGS84;
    assertUnconvertedAncestor(
      new Referenced(Types.GEOMETRY, crs),
      new Referenced(Types.POLYGON, crs),
      new Referenced(Types.POINT, crs)
    );
  }

  @Test
  public void ancestorOfTwoNumbersIsFloating() throws Exception {
    Function<Object, Object> converter =
        assertConvertedAncestor(Types.FLOATING, Types.FLOATING, Types.INTEGER);

    assertThat(converter.apply(3.0D), equalTo(3.0D));
    assertThat(converter.apply(3L), equalTo(3.0D));
  }

  @Test
  public void ancestorOfTwoSimilarNullableTypesIsTheContainedType() throws Exception {
    assertUnconvertedAncestor(Nullable.TEXT, Nullable.TEXT, Nullable.TEXT);
    assertUnconvertedAncestor(Nullable.TEXT, Nullable.TEXT, Types.TEXT);

    Function<Object, Object> convert =  assertConvertedAncestor(Nullable.FLOATING, Types.FLOATING, Nullable.INTEGER);

    // null safe
    assertThat(convert.apply(null), nullValue());

    // conversion works
    assertThat(convert.apply(3L), equalTo(3.0D));
  }

  @Test
  public void ancestorOfTwoNullableTypesWithWrappedSimpleTypesWorks() throws Exception {
    assertUnconvertedAncestor(
      Nullable.TEXT,
      Nullable.of(new WithinSet(Types.TEXT, "foo")),
      Nullable.of(new WithinSet(Types.TEXT, "bar"))
    );
  }

  @Test
  public void ancestorOfAContainingTypedAndTheContainedTypeIsAnything() throws Exception {
    assertUnconvertedAncestor(
      Nullable.TEXT,
      Nullable.of(Types.TEXT),
      Types.TEXT
    );

    assertUnconvertedAncestor(
      Types.ANYTHING,
      RSList.create(Types.TEXT),
      Types.TEXT
    );
  }

  @Test
  public void ancestorOfSimpleTypeAndSingleMemberStructIsAnything() throws Exception {
    // this requires coercion
    assertUnconvertedAncestor(
      Types.ANYTHING,
      Types.TEXT,
      Types.TEXT.asStruct()
    );
  }

  @Test
  public void canComputeMixedGeometryTypes() throws Exception {
    assertUnconvertedAncestor(Types.GEOMETRY, Types.LINE, Types.POINT);
  }

  @Test
  public void canComputeCommonListMembers() {
    RSList floatingList = RSList.create(Types.FLOATING);
    RSList integerList = RSList.create(Types.INTEGER);

    AncestorType.Converter converter = assertConvertedAncestor(floatingList, floatingList, integerList);
    assertThat(
      converter.apply(Arrays.asList(1L, 2L, 3L)),
      equalTo(Arrays.asList(1D, 2D, 3D))
    );
  }


  @Test
  public void canComputeCommonTypeFromLists() throws Exception {
    // base cases -
    assertAncestors(
      Arrays.asList(),
      Types.NOTHING
    );
    assertAncestors(
      Arrays.asList(Types.INTEGER),
      Types.INTEGER
    );
    // one lot of folding, but same types
    assertAncestors(
      Arrays.asList(Types.INTEGER, Types.INTEGER),
      Types.INTEGER
    );
    // two lots of folding, but same types (shouldn't need to fold tho)
    assertAncestors(
      Arrays.asList(Types.INTEGER, Types.INTEGER, Types.INTEGER),
      Types.INTEGER
    );
    // mixed types, but no conversion
    assertAncestors(
      Arrays.asList(Nullable.INTEGER, Types.INTEGER),
      Nullable.INTEGER
    );

    // mixed types and conversion
    assertAncestors(
      Arrays.asList(Types.INTEGER, Types.FLOATING),
      Types.FLOATING,
      Arrays.asList(1L, 2D),
      Arrays.asList(1D, 2D)
    );

    // three types and conversion
    assertAncestors(
      Arrays.asList(Types.INTEGER, Types.FLOATING, Nullable.INTEGER),
      Nullable.FLOATING,
      Arrays.asList(1L, 2D, 3L),
      Arrays.asList(1D, 2D, 3D)
    );
    // three card monty (check the order doesn't affect it)
    assertAncestors(
      Arrays.asList(Types.FLOATING, Types.INTEGER, Nullable.INTEGER),
      Nullable.FLOATING,
      Arrays.asList(1D, 2L, 3L),
      Arrays.asList(1D, 2D, 3D)
    );
    assertAncestors(
      Arrays.asList(Types.FLOATING, Nullable.INTEGER, Types.INTEGER),
      Nullable.FLOATING,
      Arrays.asList(1D, 2L, 3L),
      Arrays.asList(1D, 2D, 3D)
    );
  }

  private void assertAncestors(
      List<Type> sourceTypes,
      Type expectedAncestorType
  ) {
    assertAncestors(sourceTypes, expectedAncestorType, null, null);
  }

  private void assertAncestors(
      List<Type> sourceTypes,
      Type expectedAncestorType,
      List<Object> sourceList,
      List<Double> ancestorList
  ) {
    AncestorTypeList ancestorTypeList = typeSet.computeAncestors(sourceTypes);

    assertThat(ancestorTypeList, hasProperty("type", equalTo(expectedAncestorType)));

    if (ancestorList != null) {
      assertThat(ancestorTypeList.getConverter().apply(sourceList), equalTo(ancestorList));
    } else {
      assertThat(ancestorTypeList.getConverter(), equalTo(AncestorTypeList.IDENTITY));
    }
  }
}
