/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.types;

import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import java.util.Optional;

import org.apache.commons.lang3.tuple.Pair;
import org.junit.Test;

import nz.org.riskscape.engine.typeset.TypeSet;

public class WithMetadataTest {

  Type subject;
  TypeSet typeSet = new TypeSet();

  @Test
  public void canWrapTypeWithMetadata() {
    Pair<String, String> metadata = Pair.of("cool", "story");
    subject = WithMetadata.wrap(Types.TEXT, metadata);
    assertThat(WithMetadata.find(subject, Pair.class), is(Optional.of(metadata)));
    assertThat(WithMetadata.find(subject, String.class), is(Optional.empty()));
    assertThat(typeSet.computeAncestorNoConversion(subject, Types.TEXT), is(Types.TEXT));
    assertTrue(subject.isWrapped());
    assertTrue(subject.isA(WithMetadata.class));
  }

  @Test
  public void canWrapMultipleTypesWithMetadata() {
    Pair<String, String> metadata = Pair.of("cool", "story");
    // make WithMetadata null, just for kicks
    subject = Nullable.of(WithMetadata.wrap(Types.INTEGER, metadata, "bro"));
    assertThat(WithMetadata.find(subject, Pair.class), is(Optional.of(metadata)));
    assertThat(WithMetadata.find(subject, String.class), is(Optional.of("bro")));
    assertTrue(subject.isNumeric());
    assertTrue(subject.isNullable());
  }

  @Test
  public void canWrapNullableTypeWithMetadata() {
    Pair<String, String> metadata = Pair.of("cool", "story");
    subject = WithMetadata.wrap(Nullable.TEXT, metadata);
    assertThat(WithMetadata.find(subject, Pair.class), is(Optional.of(metadata)));
    assertThat(WithMetadata.find(subject, String.class), is(Optional.empty()));
    // note that Nullable is the outermost type, so it gets stripped off first
    assertTrue(subject.isNullable());
    assertTrue(Nullable.strip(subject).isA(WithMetadata.class));
    assertThat(Nullable.strip(subject).getUnwrappedType(), is(Types.TEXT));
  }

  @Test(expected = UnsupportedOperationException.class)
  public void cannotWrapMetadataMultipleTimes() {
    // it requires a bit more thought what should happen when we try to wrap a WithMetadata
    // up inside another WithMetadata. For now, throw an assertion to flag it's not supported
    Pair<String, String> metadata = Pair.of("cool", "story");
    Type alreadyWrapped = Nullable.of(WithMetadata.wrap(Types.TEXT, metadata));
    WithMetadata.wrap(alreadyWrapped, "bro");
  }
}
