/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.types;

import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import java.util.Optional;

import org.junit.Test;

import nz.org.riskscape.engine.typeset.TypeSet;


public class TypeTest {

  TypeSet typeSet = new TypeSet();
  Type setType = new WithinSet(Types.TEXT, "cat", "dog", "bird");
  Type rangeType = new WithinRange(Types.INTEGER, 1L, 10L);

  Type structType = Struct.of("name", Types.TEXT, "value", Types.INTEGER);
  Type linkedStructType;

  public TypeTest() {
    typeSet.add("my-type", structType);
    linkedStructType = typeSet.getLinkedType("my-type");
  }

  @Test
  public void findsOwnType() {
    assertThat(Types.INTEGER.find(Integer.class), is(Optional.of(Types.INTEGER)));
    assertThat(Types.TEXT.find(Text.class), is(Optional.of(Types.TEXT)));
    assertThat(structType.find(Struct.class), is(Optional.of(structType)));
  }

  @Test
  public void findsOwnTypeInNullalbe() {
    assertThat(Nullable.INTEGER.findAllowNull(Integer.class), is(Optional.of(Types.INTEGER)));
    assertThat(Nullable.TEXT.findAllowNull(Text.class), is(Optional.of(Types.TEXT)));
    assertThat(Nullable.of(structType).findAllowNull(Struct.class), is(Optional.of(structType)));
  }

  @Test
  public void findsTypeInWrappingTypes() {
    assertThat(setType.find(Text.class), is(Optional.of(Types.TEXT)));
    assertThat(rangeType.find(Integer.class), is(Optional.of(Types.INTEGER)));
  }

  @Test
  public void findsTypeInLinkedType() {
    assertThat(linkedStructType.find(Struct.class), is(Optional.of(structType)));
  }

  @Test
  public void findReturnsEmptyWhenTypesNotFound() {
    assertThat(Types.INTEGER.find(Text.class), is(Optional.empty()));
    assertThat(rangeType.find(Text.class), is(Optional.empty()));
    assertThat(setType.find(Floating.class), is(Optional.empty()));
    assertThat(structType.find(Text.class), is(Optional.empty()));
  }

}
