/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.types;

import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;
import static org.mockito.Mockito.*;

import java.util.Optional;

import org.geotools.api.referencing.crs.CoordinateReferenceSystem;
import org.geotools.geometry.jts.ReferencedEnvelope;
import org.geotools.referencing.crs.DefaultGeographicCRS;
import org.geotools.referencing.factory.epsg.CartesianAuthorityFactory;
import org.junit.Test;
import org.locationtech.jts.geom.Envelope;

import nz.org.riskscape.engine.typeset.TypeSet;

public class ReferencedTest {

  Envelope envelope = new Envelope(0, 1, 0, 1);
  CoordinateReferenceSystem crs1 = CartesianAuthorityFactory.GENERIC_2D;
  CoordinateReferenceSystem crs2 = mock(CoordinateReferenceSystem.class);

  @Test
  public void canWrapTypeWithCRSAndEnvelope() {
    Referenced wrapped = new Referenced(Types.LINE, crs1, envelope);
    assertSame(crs1, wrapped.getCrs());
    assertSame(envelope, wrapped.getEnvelope());
    assertEquals(new ReferencedEnvelope(envelope, crs1), wrapped.getBounds());
    assertEquals(Types.LINE, wrapped.getUnderlyingType());
  }

  @Test
  public void canWrapTypeWithCRSAndNoEnvelope() {
    Referenced wrapped = new Referenced(Types.LINE, crs1);
    assertEquals(crs1, wrapped.getCrs());
    assertEquals(Types.LINE, wrapped.getUnderlyingType());
    assertNull(wrapped.getEnvelope());
    assertNull(wrapped.getBounds());
  }

  @Test(expected = NullPointerException.class)
  public void willNotWrapWithNullCRS() {
    new Referenced(Types.LINE, (CoordinateReferenceSystem)null);
  }

  @Test(expected = IllegalArgumentException.class)
  public void willNotWrapAnAlreadyReferencedType() {
    Type wrapped = new Referenced(Types.LINE, crs1);
    new Referenced(wrapped, crs2);
  }

  @Test
  public void wrappingNullableGeomMakesNullableReferenced() {
    Type referenced = Referenced.ofNullable(Nullable.GEOMETRY, crs1);

    // the nullable should be the outer ring of this type onion
    assertTrue(Nullable.is(referenced));
    Optional<Referenced> withCRS = referenced.findAllowNull(Referenced.class);
    assertTrue(withCRS.isPresent());
    assertEquals(crs1, withCRS.get().getCrs());
    assertEquals(Types.GEOMETRY, withCRS.get().getUnderlyingType());
  }

  TypeSet typeSet = new TypeSet();

  @Test
  public void canComputeReferencedAncestorWithSameCrs() {
    Referenced lineCrs1 = new Referenced(Types.LINE, crs1);
    Referenced polygonCrs1 = new Referenced(Types.POLYGON, crs1);

    // no envelope - crs kept, but underlying type is now general geometry type
    assertThat(typeSet.computeAncestorNoConversion(lineCrs1, polygonCrs1), is(new Referenced(Types.GEOMETRY, crs1)));

    // when an envelope is present, the common ancestor should have the combined envelope
    Referenced lineCrs1Env = new Referenced(Types.LINE, crs1, envelope);
    Envelope envelope2 = new Envelope(10, 11, 10, 11);
    Referenced polygonCrs1Env = new Referenced(Types.POLYGON, crs1, envelope2);
    Envelope combined = new Envelope(0, 11, 0, 11);
    assertThat(typeSet.computeAncestorNoConversion(lineCrs1Env, polygonCrs1Env),
        is(new Referenced(Types.GEOMETRY, crs1, combined)));

    // envelope + no envelope => no envelope
    assertThat(typeSet.computeAncestorNoConversion(lineCrs1Env, polygonCrs1), is(new Referenced(Types.GEOMETRY, crs1)));
    assertThat(typeSet.computeAncestorNoConversion(lineCrs1, polygonCrs1Env), is(new Referenced(Types.GEOMETRY, crs1)));
  }

  @Test
  public void canComputeReferencedAncestorWithDifferingCrs() throws Exception {
    Referenced lineCrs1 = new Referenced(Types.LINE, crs1);
    Referenced polygonCrs2 = new Referenced(Types.POLYGON, crs2);
    Referenced polygonWgs84 = new Referenced(Types.POLYGON, DefaultGeographicCRS.WGS84);
    Referenced lineWgs843D = new Referenced(Types.LINE, DefaultGeographicCRS.WGS84_3D);

    // when CRS is different, the common ancestor should have no Referenced info
    assertThat(typeSet.computeAncestorNoConversion(lineCrs1, polygonCrs2), is(Types.GEOMETRY));
    assertThat(typeSet.computeAncestorNoConversion(polygonCrs2, polygonWgs84), is(Types.POLYGON));
    assertThat(typeSet.computeAncestorNoConversion(lineCrs1, lineWgs843D), is(Types.LINE));

    // both CRS are WGS84, but they still have differences
    assertThat(typeSet.computeAncestorNoConversion(polygonWgs84, lineWgs843D), is(Types.GEOMETRY));
  }
}
